"""Tests for word cloud fuzzy matching and order-independent intent resolution."""

import pytest
from synthos.lm.wordcloud import IntentResolver, EntityExtractor, IntentScore


@pytest.fixture
def resolver():
    return IntentResolver()


@pytest.fixture
def extractor():
    return EntityExtractor()


# ═══════════════════════════════════════════════════════════════════════════════
# Order-independent intent resolution
# ═══════════════════════════════════════════════════════════════════════════════

class TestOrderIndependence:
    """Commands should work regardless of word order."""

    def test_normal_order(self, resolver):
        r = resolver.resolve("create a directory called models")
        assert r is not None
        assert r.intent == "mkdir"

    def test_reversed_order(self, resolver):
        r = resolver.resolve("models called directory a create")
        assert r is not None
        assert r.intent == "mkdir"

    def test_shuffled_python_file(self, resolver):
        r = resolver.resolve("called train.py python file write a")
        assert r is not None
        assert r.intent == "create_py"

    def test_normal_python_file(self, resolver):
        r = resolver.resolve("write a python file called train.py")
        assert r is not None
        assert r.intent == "create_py"

    def test_shell_any_order(self, resolver):
        r = resolver.resolve("deploy.sh shell script create")
        assert r is not None
        assert r.intent == "create_sh"

    def test_scaffold_any_order(self, resolver):
        r = resolver.resolve("my-app project scaffold")
        assert r is not None
        assert r.intent == "scaffold"

    def test_markdown_any_order(self, resolver):
        r = resolver.resolve("NOTES.md markdown write")
        assert r is not None
        assert r.intent == "create_md"

    def test_list_directory(self, resolver):
        r = resolver.resolve("list the files in src")
        assert r is not None
        assert r.intent == "list_dir"

    def test_delete_file(self, resolver):
        r = resolver.resolve("delete old_config.py")
        assert r is not None
        assert r.intent == "delete_file"

    def test_tree_command(self, resolver):
        r = resolver.resolve("show me the tree")
        assert r is not None
        assert r.intent == "tree"


# ═══════════════════════════════════════════════════════════════════════════════
# Word cloud scoring
# ═══════════════════════════════════════════════════════════════════════════════

class TestWordCloudScoring:
    def test_mkdir_scores_highest_for_dir(self, resolver):
        r = resolver.resolve("create a new directory called data")
        assert r is not None
        assert r.intent == "mkdir"
        assert r.score >= 1.0

    def test_python_scores_highest_for_py(self, resolver):
        r = resolver.resolve("generate a python module called utils.py")
        assert r is not None
        assert r.intent == "create_py"

    def test_shell_scores_highest_for_sh(self, resolver):
        r = resolver.resolve("make a bash script called deploy.sh")
        assert r is not None
        assert r.intent == "create_sh"

    def test_scaffold_scores_highest(self, resolver):
        r = resolver.resolve("bootstrap a new project called my-api")
        assert r is not None
        assert r.intent == "scaffold"

    def test_no_intent_for_gibberish(self, resolver):
        r = resolver.resolve("hello world how are you today")
        assert r is None

    def test_confidence_range(self, resolver):
        r = resolver.resolve("create directory models")
        assert r is not None
        assert 0.0 <= r.confidence <= 1.0

    def test_word_cloud_summary(self, resolver):
        summary = resolver.get_word_cloud_summary("create a python file called main.py")
        assert "input" in summary
        assert "tokens" in summary
        assert "cloud" in summary
        assert "resolution" in summary
        assert len(summary["cloud"]) > 0


# ═══════════════════════════════════════════════════════════════════════════════
# Entity extraction
# ═══════════════════════════════════════════════════════════════════════════════

class TestEntityExtraction:
    def test_extract_py_filename(self, extractor):
        entities = extractor.extract("create a python file called train.py")
        assert "train.py" in entities["filenames"]

    def test_extract_sh_filename(self, extractor):
        entities = extractor.extract("write deploy.sh as a shell script")
        assert "deploy.sh" in entities["filenames"]

    def test_extract_md_filename(self, extractor):
        entities = extractor.extract("generate README.md for the project")
        assert "README.md" in entities["filenames"]

    def test_extract_named_entity(self, extractor):
        entities = extractor.extract("scaffold a project called my-awesome-app")
        assert "my-awesome-app" in entities["names"]

    def test_extract_quoted(self, extractor):
        entities = extractor.extract("create a file called 'config.yaml'")
        assert "config.yaml" in entities["quoted"]

    def test_extract_path(self, extractor):
        entities = extractor.extract("read the file src/utils/helpers.py")
        assert any("src/utils/helpers.py" in p for p in entities["paths"] + entities["filenames"])

    def test_best_name_for_create_py(self, extractor):
        entities = extractor.extract("write a python file called analyzer.py")
        name = extractor.best_name(entities, "create_py")
        assert name == "analyzer.py"

    def test_best_name_for_mkdir(self, extractor):
        entities = extractor.extract("create a directory called models")
        name = extractor.best_name(entities, "mkdir")
        assert name == "models"

    def test_best_name_for_scaffold(self, extractor):
        entities = extractor.extract("scaffold a project called my-app")
        name = extractor.best_name(entities, "scaffold")
        assert name == "my-app"


# ═══════════════════════════════════════════════════════════════════════════════
# Multi-clause resolution
# ═══════════════════════════════════════════════════════════════════════════════

class TestMultiClause:
    def test_two_clauses_with_then(self, resolver):
        results = resolver.resolve_multi(
            "create a directory called lib then write a python file called lib/utils.py"
        )
        assert len(results) >= 2
        intents = [r.intent for r in results]
        assert "mkdir" in intents
        assert "create_py" in intents

    def test_two_clauses_with_semicolon(self, resolver):
        results = resolver.resolve_multi("mkdir data; write train.py python")
        assert len(results) >= 2

    def test_three_clauses(self, resolver):
        results = resolver.resolve_multi(
            "create directory models, then write model.py python file, then create shell script run.sh"
        )
        assert len(results) >= 3

    def test_single_clause_fallback(self, resolver):
        results = resolver.resolve_multi("create a directory called output")
        assert len(results) >= 1
        assert results[0].intent == "mkdir"


# ═══════════════════════════════════════════════════════════════════════════════
# Integration with BatchProcessor
# ═══════════════════════════════════════════════════════════════════════════════

class TestBatchWordCloud:
    """Verify BatchProcessor uses word cloud for order-independent parsing."""

    @pytest.fixture
    def batch(self, tmp_path):
        from synthos.tools.executor import ToolExecutor
        from synthos.tools.generators import FileGenerator
        from synthos.tools.batch import BatchProcessor
        ws = tmp_path / "ws"
        ws.mkdir()
        ex = ToolExecutor(workspace=ws, safe_mode=False)
        gen = FileGenerator(ex)
        return BatchProcessor(ex, gen), ws

    def test_reversed_order_creates_dir(self, batch):
        bp, ws = batch
        response, plan = bp.process("models folder create")
        assert plan.completed
        assert (ws / "models").is_dir()

    def test_reversed_order_creates_py(self, batch):
        bp, ws = batch
        response, plan = bp.process("train.py python file generate")
        assert plan.completed
        assert (ws / "train.py").exists()

    def test_natural_language_output(self, batch):
        bp, ws = batch
        response, plan = bp.process("make a directory called output")
        assert plan.completed
        # Output should be natural English, not markdown
        assert "done" in response.lower() or "created" in response.lower()
        assert "##" not in response  # no markdown headers

    def test_word_cloud_in_plan(self, batch):
        bp, ws = batch
        _, plan = bp.process("create a python file called app.py")
        d = plan.to_dict()
        assert "word_cloud" in d
        assert d["word_cloud"] is not None
