"""Tests for the SYNTHOS tool system — executor, filesystem, generators, batch."""

import json
import pytest
from pathlib import Path

from synthos.tools.executor import ToolExecutor, ToolCall, ToolResult, ToolStatus
from synthos.tools.filesystem import FileSystemTools
from synthos.tools.generators import FileGenerator
from synthos.tools.batch import BatchProcessor


@pytest.fixture
def workspace(tmp_path):
    """Create a temporary workspace directory."""
    ws = tmp_path / "synthos_test_ws"
    ws.mkdir()
    return ws


@pytest.fixture
def executor(workspace):
    return ToolExecutor(workspace=workspace, safe_mode=False)


@pytest.fixture
def fs(executor):
    return FileSystemTools(executor)


@pytest.fixture
def gen(executor):
    return FileGenerator(executor)


@pytest.fixture
def batch(executor, gen):
    return BatchProcessor(executor, gen)


# ═══════════════════════════════════════════════════════════════════════════════
# ToolExecutor
# ═══════════════════════════════════════════════════════════════════════════════

class TestToolExecutor:
    def test_create_directory(self, executor, workspace):
        result = executor.execute(ToolCall("create_directory", {"path": "mydir"}))
        assert result.status == ToolStatus.SUCCESS
        assert (workspace / "mydir").is_dir()

    def test_create_file(self, executor, workspace):
        result = executor.execute(ToolCall("create_file", {"path": "test.txt", "content": "hello"}))
        assert result.status == ToolStatus.SUCCESS
        assert (workspace / "test.txt").read_text() == "hello"

    def test_read_file(self, executor, workspace):
        (workspace / "data.txt").write_text("content here")
        result = executor.execute(ToolCall("read_file", {"path": "data.txt"}))
        assert result.status == ToolStatus.SUCCESS
        assert result.output == "content here"

    def test_read_nonexistent(self, executor):
        result = executor.execute(ToolCall("read_file", {"path": "nope.txt"}))
        assert result.status == ToolStatus.ERROR

    def test_append_file(self, executor, workspace):
        (workspace / "log.txt").write_text("line1\n")
        executor.execute(ToolCall("append_file", {"path": "log.txt", "content": "line2\n"}))
        assert (workspace / "log.txt").read_text() == "line1\nline2\n"

    def test_list_directory(self, executor, workspace):
        (workspace / "a.txt").touch()
        (workspace / "b.txt").touch()
        result = executor.execute(ToolCall("list_directory", {"path": "."}))
        assert result.status == ToolStatus.SUCCESS
        assert "a.txt" in result.output
        assert "b.txt" in result.output

    def test_delete_file(self, executor, workspace):
        (workspace / "gone.txt").write_text("bye")
        result = executor.execute(ToolCall("delete_file", {"path": "gone.txt"}))
        assert result.status == ToolStatus.SUCCESS
        assert not (workspace / "gone.txt").exists()

    def test_file_exists(self, executor, workspace):
        (workspace / "here.txt").touch()
        result = executor.execute(ToolCall("file_exists", {"path": "here.txt"}))
        data = json.loads(result.output)
        assert data["exists"] is True
        assert data["type"] == "file"

    def test_tree(self, executor, workspace):
        (workspace / "sub").mkdir()
        (workspace / "sub" / "f.txt").touch()
        result = executor.execute(ToolCall("tree", {"path": ".", "depth": 2}))
        assert result.status == ToolStatus.SUCCESS
        assert "sub" in result.output
        assert "f.txt" in result.output

    def test_run_shell(self, executor):
        result = executor.execute(ToolCall("run_shell", {"command": "echo hello_synthos"}))
        assert result.status == ToolStatus.SUCCESS
        assert "hello_synthos" in result.output

    def test_run_python(self, executor):
        result = executor.execute(ToolCall("run_python", {"code": "print(2+2)"}))
        assert result.status == ToolStatus.SUCCESS
        assert "4" in result.output

    def test_unknown_tool(self, executor):
        result = executor.execute(ToolCall("nonexistent", {}))
        assert result.status == ToolStatus.ERROR
        assert "Unknown tool" in result.message

    def test_batch_execute(self, executor, workspace):
        calls = [
            ToolCall("create_directory", {"path": "batch_dir"}),
            ToolCall("create_file", {"path": "batch_dir/x.txt", "content": "x"}),
            ToolCall("create_file", {"path": "batch_dir/y.txt", "content": "y"}),
        ]
        results = executor.execute_batch(calls)
        assert len(results) == 3
        assert all(r.status == ToolStatus.SUCCESS for r in results)
        assert (workspace / "batch_dir" / "x.txt").read_text() == "x"

    def test_history_tracking(self, executor):
        executor.execute(ToolCall("run_shell", {"command": "echo a"}))
        executor.execute(ToolCall("run_shell", {"command": "echo b"}))
        assert len(executor.history) == 2


# ═══════════════════════════════════════════════════════════════════════════════
# FileSystemTools
# ═══════════════════════════════════════════════════════════════════════════════

class TestFileSystemTools:
    def test_scaffold_project(self, fs, workspace):
        structure = {
            "src": {"main.py": "print('hi')", "__init__.py": ""},
            "README.md": "# Test",
        }
        results = fs.scaffold_project("myproject", structure)
        assert all(r.status == ToolStatus.SUCCESS for r in results)
        assert (workspace / "myproject" / "src" / "main.py").read_text() == "print('hi')"
        assert (workspace / "myproject" / "README.md").read_text() == "# Test"

    def test_create_files_batch(self, fs, workspace):
        files = {
            "a.py": "# a",
            "b.sh": "#!/bin/bash",
            "c.md": "# C",
        }
        results = fs.create_files_batch(files)
        assert len(results) == 3
        assert all(r.status == ToolStatus.SUCCESS for r in results)

    def test_read_directory_context(self, fs, workspace):
        (workspace / "sub").mkdir()
        (workspace / "sub" / "f.txt").touch()
        ctx = fs.read_directory_context(".")
        assert "sub" in ctx

    def test_write_and_read(self, fs):
        fs.write_file("roundtrip.txt", "hello roundtrip")
        content = fs.read_file("roundtrip.txt")
        assert content == "hello roundtrip"

    def test_exists(self, fs, workspace):
        (workspace / "exists.txt").touch()
        assert fs.exists("exists.txt")
        assert not fs.exists("nope.txt")


# ═══════════════════════════════════════════════════════════════════════════════
# FileGenerator
# ═══════════════════════════════════════════════════════════════════════════════

class TestFileGenerator:
    def test_generate_python_module(self, gen, workspace):
        result = gen.generate_python_module(
            path="mymod.py",
            title="My Module",
            description="A test module",
            imports=["import os", "import sys"],
            functions=[{"name": "greet", "params": "name: str", "returns": "str",
                        "docstring": "Greet someone", "body": 'return f"Hello {name}"'}],
            main_body='greet("world")',
        )
        assert result.status == ToolStatus.SUCCESS
        content = (workspace / "mymod.py").read_text()
        assert "My Module" in content
        assert "import os" in content
        assert "def greet" in content
        assert "def main" in content

    def test_generate_python_script(self, gen, workspace):
        result = gen.generate_python_script("script.py", "Test script", "print('ok')")
        assert result.status == ToolStatus.SUCCESS
        assert "print('ok')" in (workspace / "script.py").read_text()

    def test_generate_shell_script(self, gen, workspace):
        result = gen.generate_shell_script("run.sh", "Runner", "Runs things", "echo running")
        assert result.status == ToolStatus.SUCCESS
        content = (workspace / "run.sh").read_text()
        assert "#!/usr/bin/env bash" in content
        assert "set -euo pipefail" in content
        assert "echo running" in content

    def test_generate_markdown(self, gen, workspace):
        result = gen.generate_markdown("doc.md", "My Doc", "A description", "## Section\nContent here")
        assert result.status == ToolStatus.SUCCESS
        content = (workspace / "doc.md").read_text()
        assert "# My Doc" in content
        assert "## Section" in content

    def test_generate_readme(self, gen, workspace):
        result = gen.generate_readme("README.md", "TestProject", "A test project")
        assert result.status == ToolStatus.SUCCESS
        content = (workspace / "README.md").read_text()
        assert "TestProject" in content
        assert "## Installation" in content

    def test_generate_project(self, gen, workspace):
        results = gen.generate_project(
            name="testproj",
            description="Test project",
            modules=[{"path": "src/core.py", "title": "Core", "description": "Core module"}],
            scripts=[{"path": "scripts/build.sh", "title": "Build", "body": "echo build"}],
            docs=[{"path": "docs/GUIDE.md", "title": "Guide", "body": "## Getting Started"}],
        )
        assert all(r.status == ToolStatus.SUCCESS for r in results)
        assert (workspace / "testproj" / "src" / "core.py").exists()
        assert (workspace / "testproj" / "scripts" / "build.sh").exists()
        assert (workspace / "testproj" / "docs" / "GUIDE.md").exists()
        assert (workspace / "testproj" / "README.md").exists()


# ═══════════════════════════════════════════════════════════════════════════════
# BatchProcessor — natural language understanding
# ═══════════════════════════════════════════════════════════════════════════════

class TestBatchProcessor:
    def test_parse_mkdir(self, batch):
        intents = batch._parse_intents("create a directory called models")
        assert len(intents) >= 1
        assert intents[0][0] == "mkdir"
        assert intents[0][1]["name"] == "models"

    def test_parse_create_py(self, batch):
        intents = batch._parse_intents("write a python file called train.py")
        assert len(intents) >= 1
        assert intents[0][0] == "create_py"

    def test_parse_create_sh(self, batch):
        intents = batch._parse_intents("create a shell script called deploy.sh")
        assert len(intents) >= 1
        assert intents[0][0] == "create_sh"

    def test_parse_create_md(self, batch):
        intents = batch._parse_intents("write a markdown file called ARCHITECTURE.md")
        assert len(intents) >= 1
        assert intents[0][0] == "create_md"

    def test_parse_scaffold(self, batch):
        intents = batch._parse_intents("scaffold a project called my-app")
        assert len(intents) >= 1
        assert intents[0][0] == "scaffold"

    def test_parse_multi_task(self, batch):
        intents = batch._parse_intents(
            "create a directory called lib and then write a python file called lib/utils.py"
        )
        assert len(intents) >= 2

    def test_process_mkdir(self, batch, workspace):
        response, plan = batch.process("create a directory called test_output")
        assert plan.completed
        assert (workspace / "test_output").is_dir()
        assert "done" in response.lower() or "created" in response.lower()

    def test_process_create_py(self, batch, workspace):
        response, plan = batch.process("write a python file called processor.py")
        assert plan.completed
        assert (workspace / "processor.py").exists()
        content = (workspace / "processor.py").read_text()
        assert "def main" in content

    def test_process_create_sh(self, batch, workspace):
        response, plan = batch.process("create a shell script called setup.sh")
        assert plan.completed
        assert (workspace / "setup.sh").exists()
        content = (workspace / "setup.sh").read_text()
        assert "#!/usr/bin/env bash" in content

    def test_process_create_md(self, batch, workspace):
        response, plan = batch.process("write a markdown doc called NOTES.md")
        assert plan.completed
        assert (workspace / "NOTES.md").exists()
        content = (workspace / "NOTES.md").read_text()
        assert "# " in content

    def test_process_scaffold(self, batch, workspace):
        response, plan = batch.process("scaffold a project called myapp")
        assert plan.completed
        assert (workspace / "myapp").is_dir()
        assert (workspace / "myapp" / "src" / "main.py").exists()
        assert (workspace / "myapp" / "tests" / "test_main.py").exists()

    def test_process_help(self, batch):
        response, plan = batch.process("what can you do")
        assert "Capabilities" in response or "can" in response.lower()

    def test_process_explain(self, batch):
        response, plan = batch.process("explain how attention works")
        assert len(response) > 0
        assert plan.completed

    def test_process_multi_step(self, batch, workspace):
        response, plan = batch.process(
            "create a directory called data and then write a python file called data/loader.py"
        )
        assert plan.completed
        assert (workspace / "data").is_dir()
        assert (workspace / "data" / "loader.py").exists()

    def test_batch_process(self, batch, workspace):
        results = batch.process_batch([
            "create a directory called batch1",
            "create a directory called batch2",
            "write a python file called batch1/run.py",
        ])
        assert len(results) == 3
        assert all(plan.completed for _, plan in results)
        assert (workspace / "batch1" / "run.py").exists()

    def test_plan_serialization(self, batch, workspace):
        _, plan = batch.process("create a directory called ser_test")
        d = plan.to_dict()
        assert "request" in d
        assert "steps" in d
        assert d["completed"] is True


# ═══════════════════════════════════════════════════════════════════════════════
# Integration: SynthoLM + tools
# ═══════════════════════════════════════════════════════════════════════════════

class TestSynthoLMTools:
    def test_lm_creates_directory(self, workspace):
        from synthos.lm.engine import SynthoLM
        from synthos.utils.config import SynthosConfig
        from synthos.utils.log import VerboseLevel

        lm = SynthoLM(config=SynthosConfig(verbose=0), verbose=VerboseLevel.QUIET,
                       workspace=str(workspace))
        result = lm.generate("create a directory called experiments")
        assert result.mode == "tool_action"
        assert (workspace / "experiments").is_dir()

    def test_lm_creates_python_file(self, workspace):
        from synthos.lm.engine import SynthoLM
        from synthos.utils.config import SynthosConfig
        from synthos.utils.log import VerboseLevel

        lm = SynthoLM(config=SynthosConfig(verbose=0), verbose=VerboseLevel.QUIET,
                       workspace=str(workspace))
        result = lm.generate("write a python file called analyzer.py")
        assert result.mode == "tool_action"
        assert (workspace / "analyzer.py").exists()

    def test_lm_plain_english_still_works(self, workspace):
        from synthos.lm.engine import SynthoLM
        from synthos.utils.config import SynthosConfig
        from synthos.utils.log import VerboseLevel

        lm = SynthoLM(config=SynthosConfig(verbose=0), verbose=VerboseLevel.QUIET,
                       workspace=str(workspace))
        # This should NOT trigger tool actions — just reasoning
        result = lm.generate("what is SYNTHOS")
        assert result.mode == "pure_syntax"
        assert len(result.text) > 0
