"""Tests for Universal Task Planner — approval, parallel execution, unknown tasks."""

import pytest
from pathlib import Path

from synthos.tools.executor import ToolExecutor, ToolStatus
from synthos.tools.generators import FileGenerator
from synthos.tools.planner import (
    TaskPlanner, ExecutionPlan, PlanStep, StepStatus,
)


@pytest.fixture
def workspace(tmp_path):
    ws = tmp_path / "planner_ws"
    ws.mkdir()
    return ws


@pytest.fixture
def planner(workspace):
    ex = ToolExecutor(workspace=workspace, safe_mode=False)
    gen = FileGenerator(ex)
    return TaskPlanner(ex, gen)


# ═══════════════════════════════════════════════════════════════════════════════
# Plan generation
# ═══════════════════════════════════════════════════════════════════════════════

class TestPlanGeneration:
    def test_plan_mkdir(self, planner):
        plan = planner.plan("create a directory called data")
        assert len(plan.steps) >= 1
        assert any(s.action is not None for s in plan.steps)

    def test_plan_python_file(self, planner):
        plan = planner.plan("write a python file called train.py")
        assert len(plan.steps) >= 1
        assert any(s.action and s.action.name == "create_file" for s in plan.steps)

    def test_plan_scaffold(self, planner):
        plan = planner.plan("scaffold a project called my-app")
        assert len(plan.steps) >= 1

    def test_plan_complex_project(self, planner):
        plan = planner.plan("set up a python project called ml-pipeline")
        assert len(plan.steps) >= 3  # multiple dirs + files

    def test_plan_multi_clause(self, planner):
        plan = planner.plan("create directory models then write model.py python file")
        assert len(plan.steps) >= 2

    def test_plan_unknown_request_with_files(self, planner):
        plan = planner.plan("I need config.yaml and setup.py for my project")
        # Should detect file-like tokens and create them
        assert len(plan.steps) >= 1

    def test_plan_pure_reasoning(self, planner):
        plan = planner.plan("what is the meaning of life")
        # Should have steps but no actions
        has_actions = any(s.action is not None for s in plan.steps)
        assert not has_actions

    def test_plan_to_dict(self, planner):
        plan = planner.plan("create directory test")
        d = plan.to_dict()
        assert "request" in d
        assert "steps" in d
        assert "parallel_groups" in d


# ═══════════════════════════════════════════════════════════════════════════════
# Parallel groups
# ═══════════════════════════════════════════════════════════════════════════════

class TestParallelGroups:
    def test_complex_project_has_parallel_groups(self, planner):
        plan = planner.plan("set up a python project called webapp")
        groups = plan.parallel_groups
        assert len(groups) >= 2  # at least root dir (group 0) and sub-dirs (group 1)

    def test_independent_steps_same_group(self, planner):
        plan = planner.plan("set up a python project called mylib")
        groups = plan.parallel_groups
        # Group 1 should have multiple sub-directories
        if 1 in groups:
            assert len(groups[1]) >= 2

    def test_dependent_steps_different_groups(self, planner):
        plan = planner.plan("set up a python project called svc")
        # Steps with dependencies should be in later groups
        for step in plan.steps:
            if step.depends_on:
                for dep_id in step.depends_on:
                    dep_step = next(s for s in plan.steps if s.id == dep_id)
                    assert step.parallel_group > dep_step.parallel_group or step.parallel_group == dep_step.parallel_group


# ═══════════════════════════════════════════════════════════════════════════════
# Approval system
# ═══════════════════════════════════════════════════════════════════════════════

class TestApproval:
    def test_approval_text_generated(self, planner):
        plan = planner.plan("set up a python project called demo")
        text = plan.approval_text()
        assert "plan" in text.lower() or "step" in text.lower()
        assert "proceed" in text.lower() or "yes" in text.lower()

    def test_needs_approval_multi_step(self, planner):
        plan = planner.plan("set up a python project called demo")
        assert plan.needs_approval  # multi-step always needs approval

    def test_destructive_needs_approval(self, planner):
        plan = planner.plan("delete old_file.txt")
        if any(s.is_destructive for s in plan.steps):
            assert plan.needs_approval

    def test_callback_approval_yes(self, planner, workspace):
        approved_plans = []
        def approve_fn(plan):
            approved_plans.append(plan)
            return True

        planner.set_approval_callback(approve_fn)
        response, plan = planner.plan_and_execute("create a directory called approved_dir")
        assert plan.approved
        assert plan.executed
        assert (workspace / "approved_dir").is_dir()

    def test_callback_approval_no(self, planner, workspace):
        def reject_fn(plan):
            return False

        planner.set_approval_callback(reject_fn)
        response, plan = planner.plan_and_execute("create a directory called rejected_dir")
        assert not plan.executed
        assert "cancelled" in response.lower() or "no actions" in response.lower()
        assert not (workspace / "rejected_dir").exists()

    def test_auto_approve(self, planner, workspace):
        response, plan = planner.plan_and_execute(
            "create a directory called auto_dir", auto_approve=True
        )
        assert plan.approved
        assert plan.executed
        assert (workspace / "auto_dir").is_dir()


# ═══════════════════════════════════════════════════════════════════════════════
# Execution
# ═══════════════════════════════════════════════════════════════════════════════

class TestExecution:
    def test_execute_single_step(self, planner, workspace):
        response, plan = planner.plan_and_execute(
            "create directory output", auto_approve=True
        )
        assert (workspace / "output").is_dir()
        assert "done" in response.lower() or "completed" in response.lower()

    def test_execute_python_file(self, planner, workspace):
        response, plan = planner.plan_and_execute(
            "write a python file called runner.py", auto_approve=True
        )
        assert (workspace / "runner.py").exists()
        content = (workspace / "runner.py").read_text()
        assert "def main" in content or "runner" in content.lower()

    def test_execute_multi_step(self, planner, workspace):
        response, plan = planner.plan_and_execute(
            "create directory src then write a python file called src/app.py",
            auto_approve=True,
        )
        assert (workspace / "src").is_dir()
        assert (workspace / "src" / "app.py").exists()

    def test_execute_complex_project(self, planner, workspace):
        response, plan = planner.plan_and_execute(
            "set up a python project called myproj", auto_approve=True
        )
        assert (workspace / "myproj").is_dir()
        assert (workspace / "myproj" / "src").is_dir()
        assert (workspace / "myproj" / "tests").is_dir()
        assert (workspace / "myproj" / "README.md").exists()

    def test_execute_reports_success(self, planner, workspace):
        response, plan = planner.plan_and_execute(
            "create directory logs", auto_approve=True
        )
        assert "done" in response.lower() or "success" in response.lower() or "completed" in response.lower()

    def test_step_statuses_updated(self, planner, workspace):
        _, plan = planner.plan_and_execute(
            "create directory status_test", auto_approve=True
        )
        for step in plan.steps:
            if step.action:
                assert step.status in (StepStatus.DONE, StepStatus.FAILED)


# ═══════════════════════════════════════════════════════════════════════════════
# Parallel execution
# ═══════════════════════════════════════════════════════════════════════════════

class TestParallelExecution:
    def test_parallel_execution(self, planner, workspace):
        """Complex project should use parallel execution for independent dirs."""
        response, plan = planner.plan_and_execute(
            "set up a python project called parallel_test", auto_approve=True
        )
        assert plan.executed
        # Check all dirs created
        assert (workspace / "parallel_test").is_dir()
        assert (workspace / "parallel_test" / "src").is_dir()
        assert (workspace / "parallel_test" / "tests").is_dir()

    def test_progress_callback(self, planner, workspace):
        progress_log = []
        def on_progress(step, msg):
            progress_log.append((step.id, msg))

        planner.set_progress_callback(on_progress)
        planner.plan_and_execute("create directory progress_test", auto_approve=True)
        # Should have at least one progress call
        assert len(progress_log) >= 1


# ═══════════════════════════════════════════════════════════════════════════════
# Order-independent (via word cloud)
# ═══════════════════════════════════════════════════════════════════════════════

class TestOrderIndependent:
    def test_reversed_order_executes(self, planner, workspace):
        response, plan = planner.plan_and_execute(
            "data folder create", auto_approve=True
        )
        assert (workspace / "data").is_dir()

    def test_shuffled_file_creates(self, planner, workspace):
        response, plan = planner.plan_and_execute(
            "train.py python file generate", auto_approve=True
        )
        assert (workspace / "train.py").exists()


# ═══════════════════════════════════════════════════════════════════════════════
# LM integration
# ═══════════════════════════════════════════════════════════════════════════════

class TestLMIntegration:
    def test_lm_uses_planner(self, workspace):
        from synthos.lm.engine import SynthoLM
        from synthos.utils.config import SynthosConfig
        from synthos.utils.log import VerboseLevel

        lm = SynthoLM(
            config=SynthosConfig(verbose=0),
            verbose=VerboseLevel.QUIET,
            workspace=str(workspace),
        )
        lm._auto_approve = True

        result = lm.generate("create a directory called lm_test")
        assert (workspace / "lm_test").is_dir()
        assert result.mode == "tool_action"

    def test_lm_reasoning_still_works(self, workspace):
        from synthos.lm.engine import SynthoLM
        from synthos.utils.config import SynthosConfig
        from synthos.utils.log import VerboseLevel

        lm = SynthoLM(
            config=SynthosConfig(verbose=0),
            verbose=VerboseLevel.QUIET,
            workspace=str(workspace),
        )
        result = lm.generate("what is SYNTHOS")
        assert result.mode != "tool_action"
        assert "SYNTHOS" in result.text
