"""Tests for SynthoLM — the syntax-based language model engine."""

import pytest
from synthos.lm.engine import SynthoLM, GenerationConfig, ModelMode, GenerationResult
from synthos.utils.config import SynthosConfig
from synthos.utils.log import VerboseLevel


@pytest.fixture
def lm():
    cfg = SynthosConfig(verbose=0)
    return SynthoLM(config=cfg, verbose=VerboseLevel.QUIET)


class TestGeneration:
    def test_basic_generation(self, lm):
        result = lm.generate("hello world")
        assert isinstance(result, GenerationResult)
        assert len(result.text) > 0
        assert result.mode == "pure_syntax"
        assert result.tokens_generated > 0
        assert result.latency_ms > 0

    def test_explain_intent(self, lm):
        result = lm.generate("what is SYNTHOS")
        assert "SYNTHOS" in result.text or "syntax" in result.text.lower()

    def test_define_command(self, lm):
        result = lm.generate("DEFINE model AS transformer")
        assert "model" in result.text.lower()
        assert "transformer" in result.text.lower()

    def test_relation_input(self, lm):
        result = lm.generate("NeuralNetwork ⊂ DeepLearning")
        assert "concept" in result.text.lower() or "relation" in result.text.lower()

    def test_list_intent(self, lm):
        result = lm.generate("list the SYNTHOS layers")
        assert len(result.text) > 0

    def test_code_intent(self, lm):
        result = lm.generate("write code for a parser")
        assert "def" in result.text or "code" in result.text.lower()

    def test_emit_command(self, lm):
        result = lm.generate("EMIT RESPONSE TYPE:EXPLAIN SUBJECT:attention DEPTH:2")
        assert len(result.text) > 0

    def test_verbose_generation(self, lm):
        cfg = GenerationConfig(verbose=True)
        result = lm.generate("explain attention", cfg)
        assert "Pipeline Trace:" in result.text
        assert "L0 LPE" in result.text


class TestPipelineTrace:
    def test_trace_keys(self, lm):
        result = lm.generate("explain how reasoning works")
        trace = result.pipeline_trace
        for key in ("lpe", "gpl", "scm", "tam", "rge", "scf", "ops"):
            assert key in trace, f"Missing trace key: {key}"

    def test_lpe_primitives(self, lm):
        result = lm.generate("hello world")
        assert result.pipeline_trace["lpe"]["primitives_matched"] > 0

    def test_scm_concepts(self, lm):
        result = lm.generate("NeuralNetwork DeepLearning")
        assert result.pipeline_trace["scm"]["concepts"] >= 1


class TestConversation:
    def test_context_grows(self, lm):
        assert lm.turn_count == 0
        lm.generate("first message")
        assert lm.turn_count == 1
        assert len(lm.context) == 2  # user + assistant

    def test_multi_turn(self, lm):
        lm.generate("hello")
        lm.generate("what is attention")
        assert lm.turn_count == 2
        assert len(lm.context) == 4

    def test_chat_convenience(self, lm):
        text = lm.chat("hello")
        assert isinstance(text, str)
        assert len(text) > 0


class TestResultSerialization:
    def test_to_dict(self, lm):
        result = lm.generate("test")
        d = result.to_dict()
        assert "text" in d
        assert "mode" in d
        assert "tokens_generated" in d
        assert "latency_ms" in d
        assert "pipeline_trace" in d
