"""Tests for Response Composer — intelligent NL + ASCII output generation."""

import pytest

from synthos.lm.composer import (
    ResponseComposer, ascii_box, ascii_tree, ascii_table, ascii_progress, _insert_path,
)
from synthos.lm.distill import KnowledgeStore


@pytest.fixture
def composer():
    return ResponseComposer()


@pytest.fixture
def knowledge():
    return KnowledgeStore()


# ═══════════════════════════════════════════════════════════════════════════════
# ASCII formatting primitives
# ═══════════════════════════════════════════════════════════════════════════════

class TestAsciiBox:
    def test_basic_box(self):
        result = ascii_box("Title", "Hello world")
        assert "╔" in result
        assert "╚" in result
        assert "Title" in result
        assert "Hello world" in result

    def test_multiline_body(self):
        result = ascii_box("Test", "Line 1\nLine 2\nLine 3")
        assert "Line 1" in result
        assert "Line 2" in result
        assert "Line 3" in result

    def test_custom_width(self):
        result = ascii_box("X", "Y", width=40)
        lines = result.split("\n")
        # Top border should be 40 chars
        assert len(lines[0]) == 40


class TestAsciiTree:
    def test_simple_tree(self):
        structure = {
            "src": {"main.py": "", "__init__.py": ""},
            "README.md": "",
        }
        result = ascii_tree("myproject", structure)
        assert "myproject" in result
        assert "src" in result
        assert "main.py" in result
        assert "README.md" in result

    def test_nested_tree(self):
        structure = {
            "src": {
                "models": {"train.py": ""},
                "utils": {"helpers.py": ""},
            },
        }
        result = ascii_tree("app", structure)
        assert "models" in result
        assert "train.py" in result
        assert "helpers.py" in result

    def test_has_connectors(self):
        structure = {"a.py": "", "b.py": ""}
        result = ascii_tree("root", structure)
        assert "├──" in result or "└──" in result


class TestAsciiTable:
    def test_basic_table(self):
        result = ascii_table(["Name", "Value"], [["foo", "1"], ["bar", "2"]])
        assert "Name" in result
        assert "foo" in result
        assert "bar" in result
        assert "┌" in result
        assert "└" in result

    def test_custom_widths(self):
        result = ascii_table(["A", "B"], [["x", "y"]], col_widths=[10, 10])
        assert "A" in result


class TestAsciiProgress:
    def test_basic_progress(self):
        items = [("Step 1", "done"), ("Step 2", "running"), ("Step 3", "pending")]
        result = ascii_progress(items)
        assert "✓" in result
        assert "►" in result
        assert "○" in result

    def test_all_done(self):
        items = [("A", "done"), ("B", "done")]
        result = ascii_progress(items)
        assert result.count("✓") == 2

    def test_failed_step(self):
        items = [("A", "done"), ("B", "failed")]
        result = ascii_progress(items)
        assert "✗" in result


# ═══════════════════════════════════════════════════════════════════════════════
# Response Composer — intelligent NL generation
# ═══════════════════════════════════════════════════════════════════════════════

class TestComposeExplain:
    def test_explain_known_topic(self, composer):
        trace = {"lpe": {"primitives_matched": 5}, "scm": {"concepts": 2}, "scf": {"coherence": 0.85}}
        result = composer.compose_explain("SYNTHOS", trace)
        # Should contain actual knowledge, not just template fill
        assert "SYNTHOS" in result
        assert "regex" in result.lower() or "syntax" in result.lower() or "layer" in result.lower()
        assert len(result) > 50  # not a stub

    def test_explain_attention(self, composer):
        trace = {"lpe": {"primitives_matched": 3}, "scm": {"concepts": 1}, "scf": {"coherence": 0.7}}
        result = composer.compose_explain("attention mechanism", trace)
        assert "attention" in result.lower()
        assert len(result) > 50

    def test_explain_unknown_topic(self, composer):
        trace = {"lpe": {"primitives_matched": 1}, "scm": {"concepts": 0}, "scf": {"coherence": 0.3}}
        result = composer.compose_explain("quantum teleportation", trace)
        # Should still produce a coherent response
        assert len(result) > 20
        assert "quantum" in result.lower() or "analyzed" in result.lower() or "pipeline" in result.lower()

    def test_explain_includes_reasoning(self, composer):
        trace = {"lpe": {"primitives_matched": 4}, "scm": {"concepts": 2}, "scf": {"coherence": 0.9}}
        result = composer.compose_explain("encryption", trace)
        assert "Reasoning:" in result or "cipher" in result.lower() or "STC" in result

    def test_explain_includes_analysis(self, composer):
        trace = {"lpe": {"primitives_matched": 12}, "scm": {"concepts": 3}, "scf": {"coherence": 0.95}}
        result = composer.compose_explain("pipeline processing", trace)
        assert "Analysis" in result or "pipeline" in result.lower()


class TestComposeCompare:
    def test_compare_known(self, composer):
        result = composer.compose_compare("attention vs encryption", "")
        assert "attention" in result.lower()
        assert "encryption" in result.lower() or "cipher" in result.lower()

    def test_compare_has_table(self, composer):
        result = composer.compose_compare("regex vs neural", "context info")
        assert "┌" in result or "│" in result  # ASCII table present


class TestComposeList:
    def test_list_items(self, composer):
        result = composer.compose_list("attention, memory, grammar", "")
        assert "1." in result
        assert "attention" in result
        assert "memory" in result

    def test_list_with_knowledge(self, composer):
        result = composer.compose_list("SYNTHOS layers", "The 7-layer pipeline")
        assert "SYNTHOS" in result


class TestComposeSummary:
    def test_summary_known(self, composer):
        result = composer.compose_summary("SYNTHOS pipeline", "")
        assert "SYNTHOS" in result or "pipeline" in result
        assert len(result) > 30

    def test_summary_with_key_points(self, composer):
        result = composer.compose_summary("encryption", "")
        # Should have key points if knowledge exists
        assert "•" in result or "cipher" in result.lower()


class TestComposeActionResult:
    def test_single_step_result(self, composer):
        plan_data = {
            "request": "create a directory called data",
            "steps": [{"description": "Create directory 'data'", "status": "success"}],
        }
        result = composer.compose_action_result(plan_data)
        assert "data" in result

    def test_multi_step_with_tree(self, composer):
        plan_data = {
            "request": "set up a project",
            "steps": [
                {"description": "Create src/", "status": "success"},
                {"description": "Create tests/", "status": "success"},
                {"description": "Create README.md", "status": "success"},
            ],
        }
        result = composer.compose_action_result(
            plan_data,
            created_dirs=["proj/src", "proj/tests"],
            created_files=["proj/README.md"],
        )
        assert "src" in result
        assert "README" in result


# ═══════════════════════════════════════════════════════════════════════════════
# End-to-end: LM produces intelligent output
# ═══════════════════════════════════════════════════════════════════════════════

class TestLMOutput:
    @pytest.fixture
    def lm(self, tmp_path):
        from synthos.lm.engine import SynthoLM
        from synthos.utils.config import SynthosConfig
        from synthos.utils.log import VerboseLevel
        ws = tmp_path / "ws"
        ws.mkdir()
        return SynthoLM(
            config=SynthosConfig(verbose=0),
            verbose=VerboseLevel.QUIET,
            workspace=str(ws),
        )

    def test_explain_produces_knowledge(self, lm):
        result = lm.generate("explain how attention works in SYNTHOS")
        # Should contain real knowledge, not just "processed through pipeline"
        text = result.text
        assert "attention" in text.lower()
        assert len(text) > 80
        # Should NOT just say "processed through 7-layer pipeline"
        assert "processed '" not in text.lower()

    def test_explain_encryption(self, lm):
        result = lm.generate("what is the STC cipher")
        text = result.text
        assert "cipher" in text.lower() or "STC" in text or "encryption" in text.lower()

    def test_code_has_ascii_box(self, lm):
        result = lm.generate("CODE attention mechanism")
        text = result.text
        assert "╔" in text  # ASCII box
        assert "def " in text  # actual code

    def test_tool_action_produces_nl(self, lm, tmp_path):
        lm._auto_approve = True
        result = lm.generate("create a directory called output")
        text = result.text
        assert "done" in text.lower() or "create" in text.lower()
        # Should be natural language, not debug output
        assert "ToolResult" not in text
