"""Tests for the SYNTHOS assistant engine and pipeline."""

import pytest
from synthos.assistant import SynthosAssistant, PipelineTrace
from synthos.utils.config import SynthosConfig
from synthos.utils.log import VerboseLevel


@pytest.fixture
def assistant():
    cfg = SynthosConfig(verbose=0)
    return SynthosAssistant(config=cfg, verbose=VerboseLevel.QUIET)


class TestAssistantCommands:
    """Test /command dispatch."""

    def test_help_command(self, assistant):
        resp = assistant.process("/help")
        assert "SYNTHOS Commands" in resp

    def test_status_command(self, assistant):
        resp = assistant.process("/status")
        assert "System Status" in resp

    def test_layers_command(self, assistant):
        resp = assistant.process("/layers")
        assert "LPE" in resp
        assert "OPS" in resp

    def test_verbose_command(self, assistant):
        resp = assistant.process("/verbose 2")
        assert "VERBOSE" in resp

    def test_unknown_command(self, assistant):
        resp = assistant.process("/nonexistent")
        assert "Unknown command" in resp

    def test_encrypt_decrypt_roundtrip(self, assistant):
        enc_resp = assistant.process("/encrypt hello world")
        # Extract hex line
        for line in enc_resp.splitlines():
            if line.startswith("Hex"):
                hex_ct = line.split(":", 1)[1].strip()
                break
        else:
            pytest.fail("No hex output found in encrypt response")

        dec_resp = assistant.process(f"/decrypt {hex_ct}")
        assert "hello world" in dec_resp

    def test_config_command(self, assistant):
        resp = assistant.process("/config")
        assert "verbose" in resp


class TestPipeline:
    """Test the 7-layer processing pipeline."""

    def test_basic_pipeline(self, assistant):
        resp = assistant.process("The quick brown fox")
        assert resp  # Non-empty response

    def test_emit_command(self, assistant):
        resp = assistant.process("EMIT RESPONSE TYPE:EXPLAIN SUBJECT:test DEPTH:1")
        assert resp

    def test_status_after_processing(self, assistant):
        assistant.process("some input")
        s = assistant.get_status()
        assert s["primitives"] > 0
        assert s["history_turns"] == 2  # user + assistant

    def test_history_grows(self, assistant):
        assert len(assistant.history) == 0
        assistant.process("one")
        assert len(assistant.history) == 2
        assistant.process("two")
        assert len(assistant.history) == 4

    def test_empty_input(self, assistant):
        resp = assistant.process("")
        assert resp == ""

    def test_episodic_memory_grows(self, assistant):
        initial = len(assistant.scf.memory_lattice.episodic_stack)
        assistant.process("first message")
        assert len(assistant.scf.memory_lattice.episodic_stack) > initial
