"""
SYNTHOS Logging Subsystem
=========================

Provides a structured, colour-aware logger that integrates with Rich for
beautiful terminal output.  Three verbosity tiers control how much internal
pipeline detail is surfaced:

    QUIET   — errors and final output only
    NORMAL  — layer transitions, summary statistics
    VERBOSE — per-round cipher traces, per-cell lattice hits, attention
              weight matrices, full parse trees
"""

from __future__ import annotations

import logging
import sys
from enum import IntEnum
from typing import Optional

from rich.console import Console
from rich.logging import RichHandler
from rich.theme import Theme

# ── Verbosity enum ─────────────────────────────────────────────────────────────

class VerboseLevel(IntEnum):
    """Verbosity tiers."""
    QUIET = 0
    NORMAL = 1
    VERBOSE = 2

# ── Theme ──────────────────────────────────────────────────────────────────────

SYNTHOS_THEME = Theme({
    "info":      "cyan",
    "warning":   "yellow",
    "error":     "bold red",
    "layer":     "bold magenta",
    "crypto":    "bold green",
    "eval":      "bold blue",
    "metric":    "bright_yellow",
    "dim":       "dim white",
    "highlight": "bold white on blue",
})

_console: Optional[Console] = None


def _get_console() -> Console:
    global _console
    if _console is None:
        _console = Console(theme=SYNTHOS_THEME, stderr=True)
    return _console


# ── Logger factory ─────────────────────────────────────────────────────────────

_configured = False


def get_logger(
    name: str = "synthos",
    level: VerboseLevel = VerboseLevel.NORMAL,
) -> logging.Logger:
    """
    Return a Rich-backed logger for *name*.

    The root ``synthos`` logger is configured once; subsequent calls just
    return child loggers.

    Parameters
    ----------
    name : str
        Dotted logger name (e.g. ``"synthos.crypto"``).
    level : VerboseLevel
        Controls both the Python log level and the detail of SYNTHOS-
        specific trace messages.
    """
    global _configured

    py_level = {
        VerboseLevel.QUIET:   logging.WARNING,
        VerboseLevel.NORMAL:  logging.INFO,
        VerboseLevel.VERBOSE: logging.DEBUG,
    }[level]

    if not _configured:
        console = _get_console()
        handler = RichHandler(
            console=console,
            show_time=True,
            show_path=False,
            rich_tracebacks=True,
            tracebacks_show_locals=(level == VerboseLevel.VERBOSE),
            markup=True,
        )
        handler.setLevel(py_level)

        root = logging.getLogger("synthos")
        root.handlers.clear()
        root.addHandler(handler)
        root.setLevel(py_level)
        _configured = True

    logger = logging.getLogger(name)
    logger.setLevel(py_level)
    return logger
