"""
SYNTHOS Configuration Management
=================================

Handles persistent configuration stored at ``~/.config/synthos/config.yaml``.
Provides typed access to all tunables: model endpoints, cipher defaults,
TUI preferences, verbosity, etc.
"""

from __future__ import annotations

import os
from dataclasses import dataclass, field, asdict
from pathlib import Path
from typing import Any, Dict, List, Optional

import yaml

# ── Defaults ───────────────────────────────────────────────────────────────────

DEFAULT_CONFIG_DIR = Path.home() / ".config" / "synthos"
DEFAULT_CONFIG_FILE = DEFAULT_CONFIG_DIR / "config.yaml"


@dataclass
class ModelEndpoint:
    """One inference backend."""
    name: str = "ollama-local"
    provider: str = "ollama"
    base_url: str = "http://localhost:11434"
    model_id: str = "llama3.1"
    ctx_len: int = 4096
    params_b: float = 8.0
    quant: str = "Q4_K_M"
    cost_per_mtok: float = 0.0


@dataclass
class CryptoConfig:
    """Encryption defaults."""
    mode: str = "cbc"          # ecb | cbc | ctr
    authenticate: bool = True
    key_file: str = str(DEFAULT_CONFIG_DIR / "stc.key")


@dataclass
class TuiConfig:
    """TUI appearance."""
    theme: str = "dark"        # dark | light | monokai
    show_layer_trace: bool = True
    animate_lattice: bool = True
    max_history: int = 500


@dataclass
class SynthosConfig:
    """
    Top-level SYNTHOS configuration.

    Attributes
    ----------
    verbose : int
        Verbosity level (0 = quiet, 1 = normal, 2 = verbose).
    models : list[ModelEndpoint]
        Registered inference backends.
    crypto : CryptoConfig
        Encryption defaults.
    tui : TuiConfig
        TUI appearance settings.
    lattice_rows : int
        Default parse-lattice row count.
    lattice_cols : int
        Default parse-lattice column count.
    attention_heads : int
        Number of attention heads in the TAM layer.
    context_window : int
        SCM context-window size (characters).
    """
    verbose: int = 1
    models: List[ModelEndpoint] = field(default_factory=lambda: [ModelEndpoint()])
    crypto: CryptoConfig = field(default_factory=CryptoConfig)
    tui: TuiConfig = field(default_factory=TuiConfig)
    lattice_rows: int = 4
    lattice_cols: int = 4
    attention_heads: int = 8
    context_window: int = 512


# ── Persistence helpers ────────────────────────────────────────────────────────

def load_config(path: Optional[Path] = None) -> SynthosConfig:
    """
    Load configuration from *path* (default ``~/.config/synthos/config.yaml``).

    If the file does not exist, returns a default ``SynthosConfig`` and
    writes it to disk so the user can customise it.
    """
    path = path or DEFAULT_CONFIG_FILE
    if path.exists():
        with open(path) as fh:
            raw: Dict[str, Any] = yaml.safe_load(fh) or {}
        return _dict_to_config(raw)
    else:
        cfg = SynthosConfig()
        save_config(cfg, path)
        return cfg


def save_config(cfg: SynthosConfig, path: Optional[Path] = None) -> None:
    """Persist *cfg* to *path*."""
    path = path or DEFAULT_CONFIG_FILE
    path.parent.mkdir(parents=True, exist_ok=True)
    with open(path, "w") as fh:
        yaml.dump(_config_to_dict(cfg), fh, default_flow_style=False, sort_keys=False)


def _config_to_dict(cfg: SynthosConfig) -> Dict[str, Any]:
    return asdict(cfg)


def _dict_to_config(d: Dict[str, Any]) -> SynthosConfig:
    models_raw = d.pop("models", [])
    crypto_raw = d.pop("crypto", {})
    tui_raw = d.pop("tui", {})

    models = [ModelEndpoint(**m) if isinstance(m, dict) else m for m in models_raw]
    crypto = CryptoConfig(**crypto_raw) if isinstance(crypto_raw, dict) else crypto_raw
    tui = TuiConfig(**tui_raw) if isinstance(tui_raw, dict) else tui_raw

    return SynthosConfig(models=models, crypto=crypto, tui=tui, **d)
