"""
SYNTHOS Rich TUI Application
==============================

A full-featured Terminal User Interface built on Textual + Rich that provides:

    - Live conversation panel with the SYNTHOS assistant
    - Real-time layer-by-layer verbose trace sidebar
    - Word cloud visualisation for intent resolution
    - Natural language in / natural language out
    - System status dashboard (primitives, lattice, concepts, memory, …)
    - Encryption / decryption panel
    - Model evaluation leaderboard viewer
    - Configurable themes and verbosity

Launch via ``synthos tui`` or ``synthos`` (default command).
"""

from __future__ import annotations

import os
import platform
import shutil
import subprocess
import sys
import time
import threading
from functools import partial

from textual.app import App, ComposeResult
from textual.worker import Worker, WorkerState
from textual.binding import Binding
from textual.containers import Container, Horizontal, Vertical, VerticalScroll
from textual.css.query import NoMatches
from textual.reactive import reactive
from textual.widgets import (
    Footer,
    Header,
    Input,
    Label,
    Markdown,
    RichLog,
    Static,
    TabbedContent,
    TabPane,
)

from rich.panel import Panel
from rich.table import Table
from rich.text import Text
from rich.syntax import Syntax
from rich.columns import Columns

from synthos.assistant import SynthosAssistant
from synthos.lm.engine import SynthoLM, GenerationConfig, GenerationResult
from synthos.utils.config import SynthosConfig, load_config
from synthos.utils.log import VerboseLevel


# ── CSS ────────────────────────────────────────────────────────────────────────

SYNTHOS_CSS = """
Screen {
    background: $surface;
}

#main-container {
    height: 1fr;
}

#chat-pane {
    width: 2fr;
    border: solid $primary;
    padding: 0 1;
}

#sidebar {
    width: 1fr;
    min-width: 36;
    border: solid $secondary;
    padding: 0 1;
}

#chat-log {
    height: 1fr;
    scrollbar-size: 1 1;
}

#layer-log {
    height: 1fr;
    scrollbar-size: 1 1;
}

#input-box {
    dock: bottom;
    height: 3;
    padding: 0 1;
}

#status-table {
    height: auto;
}

.tab-content {
    height: 1fr;
}

.section-title {
    text-style: bold;
    color: $accent;
    padding: 1 0 0 0;
}

#banner {
    text-align: center;
    color: $primary;
    padding: 1;
}

#settings-container {
    padding: 1 2;
}

.setting-row {
    height: 3;
    margin: 1 0;
}

.setting-label {
    width: 30;
    padding: 0 1;
}

.setting-value {
    width: 10;
    text-align: right;
    padding: 0 1;
    color: $accent;
}

#eval-log {
    height: 1fr;
    scrollbar-size: 1 1;
}
"""


# ── Widgets ────────────────────────────────────────────────────────────────────

class ChatLog(RichLog):
    """Scrollable rich-formatted chat history."""
    pass


class LayerLog(RichLog):
    """Sidebar log for layer-trace diagnostics."""
    pass


class EvalLog(RichLog):
    """Scrollable eval/stats log for performance metrics."""
    pass


# ── Main App ───────────────────────────────────────────────────────────────────

class SynthosApp(App):
    """
    The SYNTHOS TUI application.

    Keybindings
    -----------
    ctrl+q / q      Quit
    ctrl+t          Cycle tabs
    ctrl+v          Cycle verbosity
    ctrl+s          Show status
    """

    TITLE = "SYNTHOS v3.0"
    SUB_TITLE = "Syntactic Topological Hierarchy for Organized Symbol-based Systems"
    CSS = SYNTHOS_CSS
    ALLOW_SELECT = True

    BINDINGS = [
        Binding("ctrl+q", "quit", "Quit", show=True),
        Binding("ctrl+t", "cycle_tab", "Next Tab", show=True),
        Binding("ctrl+b", "cycle_verbose", "Verbose", show=True),
        Binding("ctrl+s", "show_status", "Status", show=True),
        Binding("ctrl+y", "copy_last", "Copy", show=True),
        Binding("ctrl+e", "show_eval", "Eval", show=True),
    ]

    verbose_label = reactive("NORMAL")

    def __init__(self, config: SynthosConfig | None = None, verbose: VerboseLevel | None = None,
                 workspace: str | None = None):
        super().__init__()
        self.config = config or load_config()
        self.vlevel = verbose if verbose is not None else VerboseLevel(self.config.verbose)
        self.assistant = SynthosAssistant(config=self.config, verbose=self.vlevel)
        # SynthoLM engine for intelligent NL+ASCII response generation
        self.lm = SynthoLM(config=self.config, verbose=self.vlevel, workspace=workspace)
        self.lm._auto_approve = True  # auto-approve tool actions in TUI
        self.verbose_label = self.vlevel.name
        self._last_response = ""  # for clipboard copy

        # Settings (adjustable via Settings tab)
        self._max_words = 4000
        self._max_search_queries = 12
        self._max_sources = 8

        # Eval history
        self._eval_runs: list = []  # list of dicts with stats per run

        # Processing state — prevents double-submit and enables async
        self._processing = False

    # ── Compose ────────────────────────────────────────────────────────────

    def compose(self) -> ComposeResult:
        yield Header()
        with TabbedContent(id="tabs"):
            with TabPane("Chat", id="tab-chat"):
                with Horizontal(id="main-container"):
                    with Vertical(id="chat-pane"):
                        yield Static(self._banner_text(), id="banner")
                        yield ChatLog(id="chat-log", highlight=True, markup=True, wrap=True)
                        yield Input(placeholder="Type a message or /help …", id="input-box")
                    with Vertical(id="sidebar"):
                        yield Static("[bold magenta]Layer Trace[/bold magenta]", classes="section-title")
                        yield LayerLog(id="layer-log", highlight=True, markup=True, wrap=True, max_lines=500)

            with TabPane("Settings", id="tab-settings"):
                yield VerticalScroll(Static(id="settings-content"), id="settings-scroll")

            with TabPane("Eval", id="tab-eval"):
                yield VerticalScroll(
                    Static(id="eval-summary"),
                    EvalLog(id="eval-log", highlight=True, markup=True, wrap=True, max_lines=1000),
                    id="eval-scroll",
                )

            with TabPane("Status", id="tab-status"):
                yield VerticalScroll(Static(id="status-content"), id="status-scroll")

            with TabPane("Crypto", id="tab-crypto"):
                yield VerticalScroll(Static(id="crypto-content"), id="crypto-scroll")

        yield Footer()

    # ── Lifecycle ──────────────────────────────────────────────────────────

    def on_mount(self) -> None:
        chat_log = self.query_one("#chat-log", ChatLog)
        chat_log.write(Text.from_markup(
            "[bold cyan]Welcome to SYNTHOS.[/bold cyan]\n"
            "Type any message to process through the 7-layer pipeline.\n"
            "Use [bold]/help[/bold] for commands.  "
            "[dim]Ctrl+Q quit  ·  Ctrl+B verbose  ·  Ctrl+Y copy  ·  Ctrl+S status  ·  Ctrl+E eval[/dim]\n"
        ))
        self._refresh_status()
        self._refresh_crypto_panel()
        self._refresh_settings_panel()
        self._refresh_eval_summary()

    # ── Input handling ─────────────────────────────────────────────────────

    def on_input_submitted(self, event: Input.Submitted) -> None:
        text = event.value.strip()
        if not text:
            return
        event.input.value = ""

        chat_log = self.query_one("#chat-log", ChatLog)

        # ── Handle slash commands (always synchronous and fast) ──
        if text.startswith("/"):
            self._handle_slash_command(text, chat_log)
            return

        # Prevent double-submit while processing
        if self._processing:
            chat_log.write(Text.from_markup(
                "[yellow]⏳ Still processing previous message — please wait…[/yellow]"
            ))
            return

        # Show user message immediately
        chat_log.write(Text.from_markup(f"\n[bold green]You ›[/bold green] {text}"))

        # Disable input and run pipeline in background worker
        self._processing = True
        input_box = self.query_one("#input-box", Input)
        input_box.placeholder = "Processing…"
        input_box.disabled = True

        # Launch async worker — keeps TUI fully interactive
        self.run_worker(
            self._process_message(text),
            name="pipeline",
            exclusive=True,
        )

    async def _process_message(self, text: str) -> None:
        """Run the full 7-layer pipeline + LM generation in a background worker."""
        chat_log = self.query_one("#chat-log", ChatLog)
        layer_log = self.query_one("#layer-log", LayerLog)

        t0 = time.perf_counter()
        response = ""
        mode = "unknown"

        try:
            layer_log.write(Text.from_markup(f"\n[bold yellow]═══ Processing ═══[/bold yellow]"))
            layer_log.write(Text.from_markup(f"[dim]Input: {text[:50]}{'…' if len(text) > 50 else ''}[/dim]"))

            # ── Word Cloud Analysis ─────────────────────────────────
            try:
                self._trace_word_cloud(text, layer_log)
            except Exception as exc:
                layer_log.write(Text.from_markup(f"  [red]Word cloud error: {exc}[/red]"))

            # ── Planner ───────────────────────────────────────────
            try:
                self._trace_planner(text, layer_log)
            except Exception as exc:
                layer_log.write(Text.from_markup(f"  [red]Planner error: {exc}[/red]"))

            # ── Layer L0: LPE ──────────────────────────────────────
            try:
                layer_log.write(Text.from_markup("[bold magenta]L0 LPE[/bold magenta] [dim]Lexical Primitive Engine[/dim]"))
                lpe_out = self.assistant._layer_lpe(text)
                n_prims = lpe_out.get("matched", 0)
                top_prims = lpe_out.get("primitives", [])[:4]
                prim_names = ", ".join(p["name"] for p in top_prims) if top_prims else "(none)"
                layer_log.write(Text.from_markup(f"  [green]✓[/green] {n_prims} primitives matched: {prim_names}"))
            except Exception as exc:
                layer_log.write(Text.from_markup(f"  [red]LPE error: {exc}[/red]"))

            # ── Layer L1: GPL ──────────────────────────────────────
            try:
                layer_log.write(Text.from_markup("[bold magenta]L1 GPL[/bold magenta] [dim]Geometric Parse Lattice[/dim]"))
                gpl_out = self.assistant._layer_gpl(text)
                gpl_path = gpl_out.get("path", [])
                gpl_ok = gpl_out.get("success", False)
                layer_log.write(Text.from_markup(
                    f"  {'[green]✓[/green]' if gpl_ok else '[red]✗[/red]'} Path: {' → '.join(gpl_path[:6]) if gpl_path else '(no traversal)'}"
                ))
            except Exception as exc:
                layer_log.write(Text.from_markup(f"  [red]GPL error: {exc}[/red]"))

            # ── Layer L2: SCM ──────────────────────────────────────
            try:
                layer_log.write(Text.from_markup("[bold magenta]L2 SCM[/bold magenta] [dim]Semantic Construct Manifold[/dim]"))
                scm_out = self.assistant._layer_scm(text)
                n_conc = scm_out.get("concepts", 0)
                n_rels = scm_out.get("relations", 0)
                layer_log.write(Text.from_markup(f"  [green]✓[/green] {n_conc} concepts, {n_rels} relations bound"))
            except Exception as exc:
                layer_log.write(Text.from_markup(f"  [red]SCM error: {exc}[/red]"))

            # ── Layer L3: TAM ──────────────────────────────────────
            tam_out = {}
            try:
                layer_log.write(Text.from_markup("[bold magenta]L3 TAM[/bold magenta] [dim]Topological Attention Mesh[/dim]"))
                tam_out = self.assistant._layer_tam(text)
                heads_used = tam_out.get("heads_used", [])
                layer_log.write(Text.from_markup(f"  [green]✓[/green] {len(heads_used)} heads fired: {', '.join(str(h) for h in heads_used[:6])}"))
            except Exception as exc:
                layer_log.write(Text.from_markup(f"  [red]TAM error: {exc}[/red]"))

            # ── Layer L4: RGE ──────────────────────────────────────
            try:
                layer_log.write(Text.from_markup("[bold magenta]L4 RGE[/bold magenta] [dim]Recursive Grammar Engine[/dim]"))
                rge_out = self.assistant._layer_rge(text)
                rules_matched = rge_out.get("matched_rules", [])
                layer_log.write(Text.from_markup(f"  [green]✓[/green] Rules: {', '.join(rules_matched) if rules_matched else '(no parse)'}"))
            except Exception as exc:
                layer_log.write(Text.from_markup(f"  [red]RGE error: {exc}[/red]"))

            # ── Layer L5: SCF ──────────────────────────────────────
            try:
                layer_log.write(Text.from_markup("[bold magenta]L5 SCF[/bold magenta] [dim]State Crystallization Field[/dim]"))
                scf_out = self.assistant._layer_scf(text, tam_out)
                coh = scf_out.get("coherence", 0.0)
                gate = scf_out.get("gate_open", False)
                mem_st = len(self.assistant.scf.memory_lattice.short_term_buffer)
                mem_lt = len(self.assistant.scf.memory_lattice.long_term_registers)
                mem_ep = len(self.assistant.scf.memory_lattice.episodic_stack)
                layer_log.write(Text.from_markup(
                    f"  [green]✓[/green] Coherence: {coh:.3f}  Gate: {'OPEN' if gate else 'CLOSED'}"
                ))
                layer_log.write(Text.from_markup(
                    f"  [dim]Memory: {mem_st} short-term · {mem_lt} long-term · {mem_ep} episodic[/dim]"
                ))
            except Exception as exc:
                layer_log.write(Text.from_markup(f"  [red]SCF error: {exc}[/red]"))

            # ── Layer L6: OPS (generate response) ────────────────────
            layer_log.write(Text.from_markup("[bold magenta]L6 OPS[/bold magenta] [dim]Output Projection Surface[/dim]"))

            # Set up agentic progress callback to show live updates in sidebar
            def _agent_cb(msg: str):
                try:
                    layer_log.write(Text.from_markup(f"  [cyan]⚡[/cyan] {msg}"))
                except Exception:
                    pass
            self.lm._agent_progress = _agent_cb

            # Route through SynthoLM engine with current settings
            gen_cfg = GenerationConfig(
                max_words=self._max_words,
                max_search_queries=self._max_search_queries,
                max_sources=self._max_sources,
            )
            result = self.lm.generate(text, config=gen_cfg)
            response = result.text
            mode = result.mode

        except Exception as exc:
            response = f"An error occurred during processing: {exc}"
            mode = "error"
            layer_log.write(Text.from_markup(f"\n  [bold red]Pipeline error: {exc}[/bold red]"))

        # ── Finalize (always runs) ─────────────────────────────────
        wall_ms = (time.perf_counter() - t0) * 1000
        word_count = len(response.split())
        char_count = len(response)
        tok_estimate = int(word_count * 1.3)
        tok_per_sec = (tok_estimate / (wall_ms / 1000)) if wall_ms > 0 else 0

        mode_colors = {"knowledge": "blue", "search": "cyan", "agentic": "yellow",
                       "tool_action": "green", "pure_syntax": "magenta", "error": "red"}
        color = mode_colors.get(mode, "white")
        layer_log.write(Text.from_markup(
            f"  [green]✓[/green] Response via [bold {color}]{mode}[/bold {color}] ({char_count} chars)"
        ))
        layer_log.write(Text.from_markup(
            f"  [dim]{word_count} words · {tok_estimate} tok · {tok_per_sec:.0f} tok/s · {wall_ms:.0f}ms[/dim]"
        ))
        layer_log.write(Text.from_markup(f"[bold yellow]─── Done in {wall_ms:.0f}ms ───[/bold yellow]"))

        # Store for clipboard
        self._last_response = response

        # Record eval stats
        run_stats = {
            "run": len(self._eval_runs) + 1,
            "prompt": text[:60],
            "mode": mode,
            "words": word_count,
            "chars": char_count,
            "tokens": tok_estimate,
            "latency_ms": round(wall_ms, 1),
            "tok_per_sec": round(tok_per_sec, 1),
            "max_words": self._max_words,
            "max_queries": self._max_search_queries,
            "max_sources": self._max_sources,
        }
        self._eval_runs.append(run_stats)
        self._log_eval_run(run_stats)
        self._refresh_eval_summary()

        # Show response in chat
        chat_log.write(Text.from_markup(f"\n[bold cyan]SYNTHOS ›[/bold cyan]"))
        for line in response.split("\n"):
            chat_log.write(Text(f"  {line}"))
        chat_log.write(Text(""))

        # Update episodic memory
        self.assistant.history.append({"role": "user", "content": text})
        self.assistant.history.append({"role": "assistant", "content": response})

        # Refresh panels
        self._refresh_status()

        # Re-enable input
        self._processing = False
        try:
            input_box = self.query_one("#input-box", Input)
            input_box.disabled = False
            input_box.placeholder = "Type a message or /help …"
            input_box.focus()
        except NoMatches:
            pass

    def _trace_planner(self, text: str, layer_log: LayerLog) -> None:
        """Show planner analysis in the sidebar."""
        from synthos.tools.planner import TaskPlanner
        from synthos.tools.executor import ToolExecutor
        from synthos.tools.generators import FileGenerator

        ex = ToolExecutor(safe_mode=True)
        gen = FileGenerator(ex)
        planner = TaskPlanner(ex, gen)
        plan = planner.plan(text)
        has_actions = any(s.action is not None for s in plan.steps)

        layer_log.write(Text.from_markup("[bold cyan]Planner[/bold cyan]"))

        if not has_actions:
            layer_log.write(Text.from_markup("  [dim]No system actions → reasoning mode[/dim]"))
        else:
            groups = plan.parallel_groups
            for gid in sorted(groups.keys()):
                group_steps = groups[gid]
                if len(group_steps) > 1:
                    layer_log.write(Text.from_markup(
                        f"  [yellow]⟐ Parallel ({len(group_steps)} concurrent)[/yellow]"
                    ))
                for step in group_steps:
                    dep = f" [dim](after {step.depends_on})[/dim]" if step.depends_on else ""
                    marker = "⚠" if step.is_destructive else "•"
                    layer_log.write(Text.from_markup(
                        f"  {marker} [white]{step.description}[/white]{dep}"
                    ))

            total = len(plan.steps)
            par = sum(1 for g in groups.values() if len(g) > 1)
            layer_log.write(Text.from_markup(
                f"  [dim]{total} steps, {par} parallel group(s)[/dim]"
            ))

        layer_log.write(Text.from_markup(""))

    def _trace_word_cloud(self, text: str, layer_log: LayerLog) -> None:
        """Show word cloud analysis in the sidebar."""
        from synthos.lm.wordcloud import IntentResolver
        resolver = IntentResolver()
        cloud = resolver.get_word_cloud_summary(text)

        layer_log.write(Text.from_markup("[bold cyan]Word Cloud[/bold cyan]"))

        # Show each significant token and its intent scores
        cloud_entries = cloud.get("cloud", [])
        if cloud_entries:
            for entry in cloud_entries[:8]:
                word = entry["word"]
                intents = entry["intents"]
                if intents:
                    top = sorted(intents.items(), key=lambda x: -x[1])[:3]
                    scores_str = "  ".join(f"[cyan]{i}[/cyan]={w:.1f}" for i, w in top)
                    layer_log.write(Text.from_markup(f"  [bold]{word}[/bold] → {scores_str}"))
        else:
            layer_log.write(Text.from_markup("  [dim](no actionable words detected)[/dim]"))

        # Show resolved intent
        resolution = cloud.get("resolution")
        if resolution:
            layer_log.write(Text.from_markup(
                f"  [bold green]Intent:[/bold green] {resolution.intent} "
                f"(score={resolution.score:.1f}, conf={resolution.confidence:.0%})"
            ))
            name = resolution.entities.get("name", "")
            if name:
                layer_log.write(Text.from_markup(f"  [bold green]Target:[/bold green] {name}"))
        else:
            layer_log.write(Text.from_markup("  [yellow]No tool intent → pipeline reasoning[/yellow]"))

        layer_log.write(Text.from_markup(""))

    # ── Actions ────────────────────────────────────────────────────────────

    def _handle_slash_command(self, text: str, chat_log: ChatLog) -> None:
        """Handle /commands in the TUI."""
        cmd = text.lower().strip()
        parts = text.strip().split()

        if cmd == "/copy":
            self.action_copy_last()
        elif cmd == "/help":
            chat_log.write(Text.from_markup(
                "\n[bold cyan]SYNTHOS Commands:[/bold cyan]\n"
                "  [green]/help[/green]           — Show this help\n"
                "  [green]/copy[/green]           — Copy last response to clipboard\n"
                "  [green]/status[/green]         — Show system status\n"
                "  [green]/eval[/green]           — Show eval panel\n"
                "  [green]/settings[/green]       — Show settings panel\n"
                "  [green]/clear[/green]          — Clear chat log\n"
                "\n"
                "[bold cyan]Settings Commands:[/bold cyan]\n"
                "  [green]/set words <N>[/green]      — Set max response words (default 4000)\n"
                "  [green]/set queries <N>[/green]    — Set search queries count (default 12)\n"
                "  [green]/set sources <N>[/green]    — Set max sources shown (default 8)\n"
                "\n"
                "[bold cyan]Keybindings:[/bold cyan]\n"
                "  [green]Ctrl+Y[/green]   — Copy last response   [green]Ctrl+E[/green]   — Show eval\n"
                "  [green]Ctrl+B[/green]   — Cycle verbosity      [green]Ctrl+S[/green]   — Show status\n"
                "  [green]Ctrl+T[/green]   — Next tab             [green]Ctrl+Q[/green]   — Quit\n"
                "\n"
                "[dim]Paste into the input box with Cmd+V (macOS) or Ctrl+Shift+V.[/dim]\n"
            ))
        elif cmd == "/status":
            self.action_show_status()
        elif cmd == "/eval":
            self.action_show_eval()
        elif cmd == "/settings":
            self._show_settings_tab()
        elif cmd == "/clear":
            chat_log.clear()
            chat_log.write(Text.from_markup("[dim]Chat cleared.[/dim]\n"))
        elif len(parts) >= 3 and parts[0].lower() == "/set":
            self._handle_set_command(parts, chat_log)
        else:
            chat_log.write(Text.from_markup(
                f"[yellow]Unknown command: {text}. Type /help for available commands.[/yellow]"
            ))

    def _handle_set_command(self, parts: list, chat_log: ChatLog) -> None:
        """Handle /set <param> <value> commands."""
        param = parts[1].lower()
        try:
            value = int(parts[2])
        except (ValueError, IndexError):
            chat_log.write(Text.from_markup("[red]Usage: /set <words|queries|sources> <number>[/red]"))
            return

        if param == "words":
            self._max_words = max(50, min(50000, value))
            chat_log.write(Text.from_markup(f"[green]✓ Max words set to {self._max_words}[/green]"))
        elif param == "queries":
            self._max_search_queries = max(1, min(20, value))
            chat_log.write(Text.from_markup(f"[green]✓ Search queries set to {self._max_search_queries}[/green]"))
        elif param == "sources":
            self._max_sources = max(1, min(20, value))
            chat_log.write(Text.from_markup(f"[green]✓ Max sources set to {self._max_sources}[/green]"))
        else:
            chat_log.write(Text.from_markup(f"[red]Unknown setting: {param}. Use words, queries, or sources.[/red]"))
            return

        self._refresh_settings_panel()

    def action_copy_last(self) -> None:
        """Copy the last SYNTHOS response to the system clipboard (cross-platform)."""
        chat_log = self.query_one("#chat-log", ChatLog)
        if not self._last_response:
            chat_log.write(Text.from_markup("[yellow]Nothing to copy yet.[/yellow]"))
            return

        copied = False
        # Try platform-native clipboard tools
        clip_cmds = []
        system = platform.system()
        if system == "Darwin":
            clip_cmds = [["pbcopy"]]
        elif system == "Linux":
            # Prefer xclip, fall back to xsel
            if shutil.which("xclip"):
                clip_cmds = [["xclip", "-selection", "clipboard"]]
            elif shutil.which("xsel"):
                clip_cmds = [["xsel", "--clipboard", "--input"]]
            elif shutil.which("wl-copy"):
                clip_cmds = [["wl-copy"]]
        elif system == "Windows":
            clip_cmds = [["clip"]]

        for cmd in clip_cmds:
            try:
                proc = subprocess.Popen(cmd, stdin=subprocess.PIPE,
                                        stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
                proc.communicate(self._last_response.encode("utf-8"), timeout=5)
                if proc.returncode == 0:
                    copied = True
                    break
            except Exception:
                continue

        # Fallback: try pyperclip
        if not copied:
            try:
                import pyperclip
                pyperclip.copy(self._last_response)
                copied = True
            except Exception:
                pass

        if copied:
            chat_log.write(Text.from_markup("[green]✓ Last response copied to clipboard.[/green]"))
        else:
            chat_log.write(Text.from_markup(
                "[yellow]Clipboard not available in this environment. "
                "Response is saved — use Ctrl+Shift+C to copy from terminal.[/yellow]"
            ))

    def action_cycle_verbose(self) -> None:
        new_level = VerboseLevel((self.vlevel.value + 1) % 3)
        self.vlevel = new_level
        self.assistant.vlevel = new_level
        self.verbose_label = new_level.name
        chat_log = self.query_one("#chat-log", ChatLog)
        chat_log.write(Text.from_markup(f"[yellow]Verbosity → {new_level.name}[/yellow]"))

    def action_show_status(self) -> None:
        self._refresh_status()
        try:
            self.query_one("#tabs", TabbedContent).active = "tab-status"
        except NoMatches:
            pass

    def action_show_eval(self) -> None:
        self._refresh_eval_summary()
        try:
            self.query_one("#tabs", TabbedContent).active = "tab-eval"
        except NoMatches:
            pass

    def _show_settings_tab(self) -> None:
        self._refresh_settings_panel()
        try:
            self.query_one("#tabs", TabbedContent).active = "tab-settings"
        except NoMatches:
            pass

    def action_cycle_tab(self) -> None:
        tabs = self.query_one("#tabs", TabbedContent)
        tab_ids = ["tab-chat", "tab-settings", "tab-eval", "tab-status", "tab-crypto"]
        current = tabs.active
        idx = tab_ids.index(current) if current in tab_ids else 0
        tabs.active = tab_ids[(idx + 1) % len(tab_ids)]

    # ── Panel refreshers ──────────────────────────────────────────────────

    def _refresh_status(self) -> None:
        try:
            widget = self.query_one("#status-content", Static)
        except NoMatches:
            return

        s = self.assistant.get_status()
        table = Table(title="SYNTHOS System Status", show_header=True, header_style="bold magenta")
        table.add_column("Layer", style="cyan", width=8)
        table.add_column("Subsystem", style="white", width=30)
        table.add_column("Value", style="green", width=20)

        table.add_row("L0", "Lexical Primitive Engine", f"{s['primitives']} primitives")
        table.add_row("L1", "Geometric Parse Lattice", f"{s['lattice_cells']} cells")
        table.add_row("L2", "Semantic Construct Manifold", f"{s['concepts']} concepts, {s['relations']} relations")
        table.add_row("L3", "Topological Attention Mesh", f"{s['attention_heads']} heads")
        table.add_row("L4", "Recursive Grammar Engine", f"{s['grammar_rules']} rules")
        table.add_row("L5", "State Crystallization Field", f"coh={s['coherence']:.3f}  mem={s['memory_st']}/{s['memory_lt']}/{s['memory_ep']}")
        table.add_row("L6", "Output Projection Surface", "active")
        table.add_row("", "Conversation turns", str(s["history_turns"]))
        table.add_row("", "Verbosity", s["verbose"])

        widget.update(table)

    def _refresh_settings_panel(self) -> None:
        """Render the Settings panel with current values."""
        try:
            widget = self.query_one("#settings-content", Static)
        except NoMatches:
            return

        from synthos.lm.agent import knowledge_count

        table = Table(title="SYNTHOS Settings", show_header=True, header_style="bold cyan",
                      show_lines=True, expand=True)
        table.add_column("Setting", style="white", width=30)
        table.add_column("Value", style="green", width=12, justify="right")
        table.add_column("Range", style="dim", width=15, justify="center")
        table.add_column("Description", style="dim", width=40)

        table.add_row(
            "Max Response Words", str(self._max_words), "50 – 50,000",
            "Maximum words in generated responses"
        )
        table.add_row(
            "Search Queries", str(self._max_search_queries), "1 – 20",
            "Number of DuckDuckGo queries for deep research"
        )
        table.add_row(
            "Max Sources", str(self._max_sources), "1 – 20",
            "Maximum sources shown in search results"
        )
        table.add_row(
            "Knowledge Store", str(knowledge_count()), "—",
            "Total facts in persistent knowledge store"
        )
        table.add_row(
            "Verbosity", self.vlevel.name, "QUIET / NORMAL / DEBUG",
            "Layer trace detail level"
        )

        help_text = Text.from_markup(
            "\n[bold cyan]How to change settings:[/bold cyan]\n\n"
            "  [green]/set words 500[/green]      — Set max response to 500 words\n"
            "  [green]/set words 4000[/green]     — Set max response to 4000 words (default)\n"
            "  [green]/set queries 6[/green]      — Use 6 search queries (faster)\n"
            "  [green]/set queries 12[/green]     — Use 12 search queries (default, thorough)\n"
            "  [green]/set queries 20[/green]     — Use 20 search queries (maximum depth)\n"
            "  [green]/set sources 3[/green]      — Show 3 sources\n"
            "  [green]/set sources 8[/green]      — Show 8 sources (default)\n"
            "\n"
            "[dim]Settings apply immediately to the next generation.[/dim]\n"
        )

        from rich.console import Group
        widget.update(Group(table, help_text))

    def _log_eval_run(self, stats: dict) -> None:
        """Append a run's stats to the eval log."""
        try:
            eval_log = self.query_one("#eval-log", EvalLog)
        except NoMatches:
            return

        mode_colors = {"knowledge": "blue", "search": "cyan", "agentic": "yellow",
                       "tool_action": "green", "pure_syntax": "magenta"}
        color = mode_colors.get(stats["mode"], "white")

        eval_log.write(Text.from_markup(
            f"\n[bold white]═══ Run #{stats['run']} ═══[/bold white]"
        ))
        eval_log.write(Text.from_markup(
            f"  [dim]Prompt:[/dim] {stats['prompt']}"
        ))
        eval_log.write(Text.from_markup(
            f"  [dim]Mode:[/dim]   [bold {color}]{stats['mode']}[/bold {color}]"
        ))
        eval_log.write(Text.from_markup(
            f"  [bold green]{stats['tok_per_sec']:,.1f} tok/s[/bold green]  ·  "
            f"{stats['tokens']} tokens  ·  {stats['words']} words  ·  {stats['chars']} chars"
        ))
        eval_log.write(Text.from_markup(
            f"  [dim]Latency: {stats['latency_ms']:.1f}ms  ·  "
            f"Settings: words={stats['max_words']} queries={stats['max_queries']} sources={stats['max_sources']}[/dim]"
        ))

    def _refresh_eval_summary(self) -> None:
        """Update the eval summary panel with aggregate stats."""
        try:
            widget = self.query_one("#eval-summary", Static)
        except NoMatches:
            return

        if not self._eval_runs:
            widget.update(Text.from_markup(
                "[bold cyan]Eval Panel[/bold cyan]\n\n"
                "[dim]No runs yet. Send a message in the Chat tab to see performance stats here.[/dim]\n"
                "[dim]Each run shows: tok/s, latency, mode, word count, and current settings.[/dim]\n"
            ))
            return

        runs = self._eval_runs
        total_runs = len(runs)
        avg_latency = sum(r["latency_ms"] for r in runs) / total_runs
        avg_tps = sum(r["tok_per_sec"] for r in runs) / total_runs
        avg_words = sum(r["words"] for r in runs) / total_runs
        total_tokens = sum(r["tokens"] for r in runs)
        best_tps = max(r["tok_per_sec"] for r in runs)
        worst_tps = min(r["tok_per_sec"] for r in runs)

        # Mode distribution
        mode_counts: dict = {}
        for r in runs:
            mode_counts[r["mode"]] = mode_counts.get(r["mode"], 0) + 1

        table = Table(title="SYNTHOS Eval Summary", show_header=True,
                      header_style="bold cyan", show_lines=True, expand=True)
        table.add_column("Metric", style="white", width=25)
        table.add_column("Value", style="green", width=20, justify="right")

        table.add_row("Total Runs", str(total_runs))
        table.add_row("Total Tokens Generated", f"{total_tokens:,}")
        table.add_row("Avg tok/s", f"{avg_tps:,.1f}")
        table.add_row("Best tok/s", f"{best_tps:,.1f}")
        table.add_row("Worst tok/s", f"{worst_tps:,.1f}")
        table.add_row("Avg Latency", f"{avg_latency:,.1f} ms")
        table.add_row("Avg Words/Response", f"{avg_words:,.0f}")

        for mode, count in sorted(mode_counts.items(), key=lambda x: -x[1]):
            pct = count / total_runs * 100
            table.add_row(f"Mode: {mode}", f"{count} ({pct:.0f}%)")

        # Current settings
        table.add_row("─── Current Settings ───", "")
        table.add_row("Max Words", str(self._max_words))
        table.add_row("Search Queries", str(self._max_search_queries))
        table.add_row("Max Sources", str(self._max_sources))

        widget.update(table)

    def _refresh_crypto_panel(self) -> None:
        try:
            widget = self.query_one("#crypto-content", Static)
        except NoMatches:
            return

        text = Text.from_markup(
            "[bold green]SYNTHOS Symbolic Topology Cipher (STC)[/bold green]\n\n"
            "The STC is a 256-bit block cipher inspired by SYNTHOS's\n"
            "pattern-geometry architecture:\n\n"
            "  [cyan]1. Lattice Permutation[/cyan]  — key-derived positional scatter\n"
            "  [cyan]2. Regex-Topology S-box[/cyan] — named-capture hash substitution\n"
            "  [cyan]3. Möbius Fold[/cyan]          — XOR-twist half-state inversion\n\n"
            f"  Rounds : {12}  ·  Block : 128 bit  ·  Key : 256 bit\n"
            f"  Mode   : STC-{self.config.crypto.mode.upper()}\n\n"
            "Use [bold]/encrypt <text>[/bold] and [bold]/decrypt <hex>[/bold]\n"
            "in the Chat tab to try it out.\n"
        )
        widget.update(text)

    # ── Helpers ────────────────────────────────────────────────────────────

    @staticmethod
    def _banner_text() -> str:
        return (
            "[bold cyan]"
            "  ____  _   _ _   _ _____ _   _  ___  ____\n"
            " / ___|| \\ | | \\ | |_   _| | | |/ _ \\/ ___|\n"
            " \\___ \\|  \\| |  \\| | | | | |_| | | | \\___ \\\n"
            "  ___) | |\\  | |\\  | | | |  _  | |_| |___) |\n"
            " |____/|_| \\_|_| \\_| |_| |_| |_|\\___/|____/\n"
            "[/bold cyan]"
            "[dim]  v3.0 · 7 Layers · Deep Research · STC Cipher[/dim]"
        )


# ── Entry point ────────────────────────────────────────────────────────────────

def run_tui(config: SynthosConfig | None = None, verbose: VerboseLevel | None = None,
            workspace: str | None = None) -> None:
    """Launch the SYNTHOS TUI."""
    app = SynthosApp(config=config, verbose=verbose, workspace=workspace)
    app.run()
