"""
Universal Task Planner — figures out how to do ANYTHING, asks for approval, executes.

Architecture:
  1. Analyze: Break any request into reasoning + actionable sub-tasks
  2. Plan: Build a dependency graph of steps (which can run in parallel)
  3. Approve: Present the plan in natural language, wait for confirmation
  4. Execute: Run independent steps in parallel, dependent steps sequentially
  5. Report: Natural language summary of results

The planner works even for requests it has never seen before by:
  - Decomposing into known primitives (file ops, shell, code gen)
  - Inferring missing details from context
  - Chaining small steps into complex workflows
"""

from __future__ import annotations

import os
import re
import time
import concurrent.futures
from dataclasses import dataclass, field
from enum import Enum
from typing import Any, Callable, Dict, List, Optional, Set, Tuple

from synthos.tools.executor import ToolExecutor, ToolCall, ToolResult, ToolStatus
from synthos.tools.generators import FileGenerator


# ═══════════════════════════════════════════════════════════════════════════════
# Data structures
# ═══════════════════════════════════════════════════════════════════════════════

class StepStatus(Enum):
    PENDING = "pending"
    APPROVED = "approved"
    RUNNING = "running"
    DONE = "done"
    FAILED = "failed"
    SKIPPED = "skipped"


@dataclass
class PlanStep:
    """One step in an execution plan."""
    id: int
    description: str           # natural language description
    reasoning: str             # why this step is needed
    action: Optional[ToolCall] = None
    depends_on: List[int] = field(default_factory=list)  # step IDs this depends on
    status: StepStatus = StepStatus.PENDING
    result: Optional[ToolResult] = None
    parallel_group: int = 0    # steps in same group can run in parallel
    is_destructive: bool = False  # requires explicit approval
    confidence: float = 1.0

    def to_dict(self) -> Dict[str, Any]:
        return {
            "id": self.id,
            "description": self.description,
            "reasoning": self.reasoning,
            "action": repr(self.action) if self.action else None,
            "depends_on": self.depends_on,
            "status": self.status.value,
            "parallel_group": self.parallel_group,
            "is_destructive": self.is_destructive,
            "confidence": self.confidence,
        }


@dataclass
class ExecutionPlan:
    """A complete plan with dependency graph and parallel groups."""
    request: str
    summary: str
    steps: List[PlanStep] = field(default_factory=list)
    approved: bool = False
    executed: bool = False

    @property
    def parallel_groups(self) -> Dict[int, List[PlanStep]]:
        """Group steps by their parallel execution group."""
        groups: Dict[int, List[PlanStep]] = {}
        for step in self.steps:
            groups.setdefault(step.parallel_group, []).append(step)
        return groups

    @property
    def needs_approval(self) -> bool:
        return any(s.is_destructive for s in self.steps) or len(self.steps) > 1

    def to_dict(self) -> Dict[str, Any]:
        return {
            "request": self.request,
            "summary": self.summary,
            "steps": [s.to_dict() for s in self.steps],
            "approved": self.approved,
            "executed": self.executed,
            "parallel_groups": {k: [s.id for s in v] for k, v in self.parallel_groups.items()},
        }

    def approval_text(self) -> str:
        """Generate a human-readable approval prompt."""
        lines = [f"I've analyzed your request and here's my plan:\n"]
        lines.append(f"  \"{self.request}\"\n")

        groups = self.parallel_groups
        group_ids = sorted(groups.keys())

        for gid in group_ids:
            group_steps = groups[gid]
            if len(group_steps) > 1:
                lines.append(f"  [Parallel Group {gid + 1}] — these {len(group_steps)} steps run simultaneously:")
            for step in group_steps:
                dep_str = ""
                if step.depends_on:
                    dep_str = f" (after step {', '.join(str(d) for d in step.depends_on)})"
                marker = "⚠" if step.is_destructive else "•"
                lines.append(f"    {marker} Step {step.id}: {step.description}{dep_str}")
                if step.reasoning:
                    lines.append(f"      Reason: {step.reasoning}")

        total = len(self.steps)
        destructive = sum(1 for s in self.steps if s.is_destructive)
        parallel = sum(1 for g in groups.values() if len(g) > 1)
        lines.append(f"\n  Total: {total} step(s), {parallel} parallel group(s)")
        if destructive:
            lines.append(f"  ⚠ {destructive} step(s) are destructive and need your approval.")
        lines.append("\nShall I proceed? (yes/no)")
        return "\n".join(lines)


# ═══════════════════════════════════════════════════════════════════════════════
# Task decomposition patterns — how to break down complex requests
# ═══════════════════════════════════════════════════════════════════════════════

# High-level task patterns that map to multi-step plans
_TASK_PATTERNS: List[Tuple[str, str, List[Dict[str, Any]]]] = [
    # (regex, task_type, step_templates)

    # "set up / create a <type> project called <name>"
    (r"(?i)(?:set\s+up|create|build|make|start)\s+(?:a\s+)?(?:(?:new|full|complete)\s+)?(?P<type>python|node|web|api|flask|django|react|ml|data|cli)\s+(?:project|app|application|package)\s+(?:called\s+|named\s+)?(?P<name>\S+)",
     "project_setup",
     [
         {"desc": "Create project root directory '{name}'", "tool": "create_directory", "args_tpl": {"path": "{name}"}, "group": 0},
         {"desc": "Create source directory '{name}/src'", "tool": "create_directory", "args_tpl": {"path": "{name}/src"}, "group": 1, "deps": [0]},
         {"desc": "Create tests directory '{name}/tests'", "tool": "create_directory", "args_tpl": {"path": "{name}/tests"}, "group": 1, "deps": [0]},
         {"desc": "Create __init__.py for source package", "tool": "create_file", "args_tpl": {"path": "{name}/src/__init__.py", "content": ""}, "group": 2, "deps": [1]},
         {"desc": "Create main module '{name}/src/main.py'", "tool": "create_file", "content_gen": "py_main", "group": 2, "deps": [1]},
         {"desc": "Create test file '{name}/tests/test_main.py'", "tool": "create_file", "content_gen": "py_test", "group": 2, "deps": [2]},
         {"desc": "Create README.md", "tool": "create_file", "content_gen": "readme", "group": 2, "deps": [0]},
     ]),

    # "organize / restructure / clean up <path>"
    (r"(?i)(?:organize|restructure|clean\s+up|sort|tidy)\s+(?:the\s+)?(?:files?\s+in\s+)?(?P<path>\S+)",
     "organize",
     [
         {"desc": "List current contents of '{path}'", "tool": "list_directory", "args_tpl": {"path": "{path}"}, "group": 0},
         {"desc": "Show directory tree for '{path}'", "tool": "tree", "args_tpl": {"path": "{path}"}, "group": 0},
     ]),

    # "deploy / publish <name>"
    (r"(?i)(?:deploy|publish|release|ship)\s+(?:the\s+)?(?:project\s+)?(?P<name>\S+)",
     "deploy",
     [
         {"desc": "Check project structure", "tool": "tree", "args_tpl": {"path": "{name}"}, "group": 0},
         {"desc": "Run tests", "tool": "run_shell", "args_tpl": {"command": "cd {name} && python -m pytest tests/ -v 2>&1 || true"}, "group": 1, "deps": [0]},
         {"desc": "Create deployment script", "tool": "create_file", "content_gen": "deploy_sh", "group": 1, "deps": [0]},
     ]),

    # "test / verify / check <project_name>" — only match path-like or project names, not abstract nouns
    (r"(?i)(?:test|verify|validate)\s+(?:the\s+)?(?:project\s+)?(?P<name>[\w./-]+(?:/[\w./-]+)+|[\w-]+(?:_[\w-]+)+|\S+\.[\w]+)",
     "test",
     [
         {"desc": "Check if '{name}' exists", "tool": "file_exists", "args_tpl": {"path": "{name}"}, "group": 0},
         {"desc": "List contents", "tool": "list_directory", "args_tpl": {"path": "{name}"}, "group": 0},
         {"desc": "Run tests if available", "tool": "run_shell", "args_tpl": {"command": "cd {name} && python -m pytest tests/ -v 2>&1 || echo 'No tests found'"}, "group": 1, "deps": [0, 1]},
     ]),

    # "back up / copy <source> to <dest>"
    (r"(?i)(?:back\s*up|copy|duplicate|clone)\s+(?:the\s+)?(?P<source>\S+)\s+(?:to|into|as)\s+(?P<dest>\S+)",
     "backup",
     [
         {"desc": "Read source '{source}'", "tool": "tree", "args_tpl": {"path": "{source}"}, "group": 0},
         {"desc": "Create backup directory '{dest}'", "tool": "create_directory", "args_tpl": {"path": "{dest}"}, "group": 0},
         {"desc": "Copy files", "tool": "run_shell", "args_tpl": {"command": "cp -r {source}/* {dest}/ 2>&1 || echo 'Copy failed'"}, "group": 1, "deps": [0, 1], "destructive": True},
     ]),

    # "install / add dependency <pkg>"
    (r"(?i)(?:install|add)\s+(?:the\s+)?(?:dependency|package|module|lib|library)\s+(?P<pkg>\S+)",
     "install",
     [
         {"desc": "Install '{pkg}' via pip", "tool": "run_shell", "args_tpl": {"command": "pip install {pkg}"}, "group": 0, "destructive": True},
     ]),

    # ── Shell command execution patterns ─────────────────────────────

    # "run / execute <command>"
    (r"(?i)(?:run|execute|exec)\s+(?:the\s+)?(?:command\s+)?['\"](?P<cmd>[^'\"]+)['\"]",
     "run_command",
     [
         {"desc": "Run command: {cmd}", "tool": "run_shell", "args_tpl": {"command": "{cmd}"}, "group": 0, "destructive": True},
     ]),

    # "run / execute <command>" (without quotes)
    (r"(?i)(?:run|execute|exec)\s+(?:the\s+)?(?:command\s+)?(?P<cmd>(?:ls|cat|echo|pwd|whoami|date|uptime|df|du|ps|top|find|grep|wc|head|tail|sort|uniq|curl|wget|ping|which|env|printenv|uname|hostname|id|free|lsof|netstat|ifconfig|ip|mount|chmod|chown|mkdir|rmdir|cp|mv|touch|tar|gzip|gunzip|zip|unzip|git|npm|pip|python|node|ruby|go|cargo|make|cmake|brew|apt|yum|docker|kubectl)\b.*)",
     "run_command",
     [
         {"desc": "Run: {cmd}", "tool": "run_shell", "args_tpl": {"command": "{cmd}"}, "group": 0, "destructive": True},
     ]),

    # "list / show files in <path>"
    (r"(?i)(?:list|show|display|view)\s+(?:the\s+)?(?:files?|contents?|items?)\s+(?:in|of|at|for)\s+(?P<path>\S+)",
     "list_files",
     [
         {"desc": "List files in '{path}'", "tool": "list_directory", "args_tpl": {"path": "{path}"}, "group": 0},
         {"desc": "Show tree for '{path}'", "tool": "tree", "args_tpl": {"path": "{path}"}, "group": 0},
     ]),

    # "show / list / check processes / running processes"
    (r"(?i)(?:show|list|check|view|display)\s+(?:the\s+)?(?:running\s+)?(?:processes|procs|tasks)",
     "show_processes",
     [
         {"desc": "List running processes", "tool": "run_shell", "args_tpl": {"command": "ps aux | head -20"}, "group": 0},
     ]),

    # "check / show disk space / disk usage"
    (r"(?i)(?:check|show|view|display)\s+(?:the\s+)?(?:disk\s+)?(?:space|usage|storage|capacity)",
     "check_disk",
     [
         {"desc": "Check disk usage", "tool": "run_shell", "args_tpl": {"command": "df -h"}, "group": 0},
     ]),

    # "find files matching / find <pattern> in <path>"
    (r"(?i)(?:find|search\s+for|locate)\s+(?:files?\s+)?(?:matching\s+|named\s+|called\s+|with\s+)?(?P<pattern>\S+)\s+in\s+(?P<path>\S+)",
     "find_files",
     [
         {"desc": "Find files matching '{pattern}' in '{path}'", "tool": "run_shell", "args_tpl": {"command": "find {path} -name '{pattern}' 2>/dev/null | head -30"}, "group": 0},
     ]),

    # "find files matching / find <pattern>" (no path = current dir)
    (r"(?i)(?:find|search\s+for|locate)\s+(?:files?\s+)?(?:matching\s+|named\s+|called\s+|with\s+)?(?P<pattern>\S+)",
     "find_files",
     [
         {"desc": "Find files matching '{pattern}'", "tool": "run_shell", "args_tpl": {"command": "find . -name '{pattern}' 2>/dev/null | head -30"}, "group": 0},
     ]),

    # "show / check system info / system status"
    (r"(?i)(?:show|check|view|display|get)\s+(?:the\s+)?(?:system\s+)?(?:info|information|status|details|specs)",
     "system_info",
     [
         {"desc": "Show system info", "tool": "run_shell", "args_tpl": {"command": "uname -a"}, "group": 0},
         {"desc": "Show disk usage", "tool": "run_shell", "args_tpl": {"command": "df -h"}, "group": 0},
         {"desc": "Show memory", "tool": "run_shell", "args_tpl": {"command": "vm_stat 2>/dev/null || free -h 2>/dev/null || echo 'Memory info not available'"}, "group": 0},
     ]),

    # "read / show / cat <file>"
    (r"(?i)(?:read|show|cat|display|view|open|print)\s+(?:the\s+)?(?:contents?\s+of\s+)?(?:file\s+)?(?P<path>\S+\.\w+)",
     "read_file",
     [
         {"desc": "Read file '{path}'", "tool": "read_file", "args_tpl": {"path": "{path}"}, "group": 0},
     ]),

    # "count lines / words in <file>"
    (r"(?i)(?:count|how\s+many)\s+(?:the\s+)?(?:lines?|words?|chars?|characters?)\s+(?:in|of)\s+(?P<path>\S+)",
     "count_file",
     [
         {"desc": "Count lines/words in '{path}'", "tool": "run_shell", "args_tpl": {"command": "wc {path}"}, "group": 0},
     ]),

    # "check / show / what's in environment / env vars"
    (r"(?i)(?:show|check|list|display|get)\s+(?:the\s+)?(?:environment|env)\s*(?:variables?|vars?)?",
     "env_vars",
     [
         {"desc": "Show environment variables", "tool": "run_shell", "args_tpl": {"command": "env | sort | head -30"}, "group": 0},
     ]),

    # "check / show / what's the current directory / pwd"
    (r"(?i)(?:show|check|what(?:'s|\s+is))\s+(?:the\s+)?(?:current|working)\s+(?:directory|dir|path|folder)",
     "pwd",
     [
         {"desc": "Show current directory", "tool": "run_shell", "args_tpl": {"command": "pwd"}, "group": 0},
     ]),

    # "ping <host>"
    (r"(?i)ping\s+(?P<host>\S+)",
     "ping",
     [
         {"desc": "Ping {host}", "tool": "run_shell", "args_tpl": {"command": "ping -c 3 {host}"}, "group": 0},
     ]),

    # "check if port <port> is open / listening"
    (r"(?i)(?:check|is)\s+(?:if\s+)?port\s+(?P<port>\d+)\s+(?:is\s+)?(?:open|listening|in\s+use|used)",
     "check_port",
     [
         {"desc": "Check port {port}", "tool": "run_shell", "args_tpl": {"command": "lsof -i :{port} 2>/dev/null || echo 'Port {port} is not in use'"}, "group": 0},
     ]),

    # "git status / git log / git <cmd>"
    (r"(?i)(?:show\s+)?git\s+(?P<cmd>status|log|diff|branch|remote|stash)",
     "git_cmd",
     [
         {"desc": "Run git {cmd}", "tool": "run_shell", "args_tpl": {"command": "git {cmd}"}, "group": 0},
     ]),
]

# Destructive actions that always need approval
_DESTRUCTIVE_TOOLS = {"run_shell", "run_python", "delete_file"}


# ═══════════════════════════════════════════════════════════════════════════════
# Content generators for plan steps
# ═══════════════════════════════════════════════════════════════════════════════

def _gen_content(gen_type: str, params: Dict[str, str]) -> Tuple[str, str]:
    """Generate (path, content) for a content_gen step."""
    name = params.get("name", "project")
    ptype = params.get("type", "python")

    if gen_type == "py_main":
        path = f"{name}/src/main.py"
        content = f'"""\n{name} — main entry point.\n"""\n\n\ndef main():\n    print("{name} is running")\n\n\nif __name__ == "__main__":\n    main()\n'
        return path, content

    if gen_type == "py_test":
        path = f"{name}/tests/test_main.py"
        content = f'"""Tests for {name}."""\nimport pytest\n\n\ndef test_main():\n    assert True  # replace with real tests\n'
        return path, content

    if gen_type == "readme":
        path = f"{name}/README.md"
        content = f"# {name}\n\nA {ptype} project generated by SYNTHOS.\n\n## Setup\n\n```bash\ncd {name}\npip install -e .\n```\n\n## Usage\n\n```bash\npython -m src.main\n```\n"
        return path, content

    if gen_type == "deploy_sh":
        path = f"{name}/deploy.sh"
        content = f"#!/usr/bin/env bash\nset -euo pipefail\necho 'Deploying {name}...'\npython -m pytest tests/ -v\necho 'All tests passed. Ready to deploy.'\n"
        return path, content

    return f"{name}/generated.txt", f"# Generated by SYNTHOS for {name}\n"


# ═══════════════════════════════════════════════════════════════════════════════
# Universal Task Planner
# ═══════════════════════════════════════════════════════════════════════════════

class TaskPlanner:
    """
    Universal task planner that can figure out how to do anything.

    Flow:
      1. plan(request) → ExecutionPlan  (analyzes, decomposes, builds plan)
      2. plan.approval_text()           (present to user)
      3. approve(plan)                  (user says yes)
      4. execute(plan) → results        (runs with parallel execution)
    """

    def __init__(self, executor: ToolExecutor, generator: FileGenerator):
        self.executor = executor
        self.generator = generator
        self._approval_callback: Optional[Callable[[ExecutionPlan], bool]] = None
        self._progress_callback: Optional[Callable[[PlanStep, str], None]] = None
        # Lazy import
        from synthos.lm.wordcloud import IntentResolver
        self.resolver = IntentResolver()

    def set_approval_callback(self, fn: Callable[[ExecutionPlan], bool]):
        """Set a callback for approval. If None, auto-approves non-destructive plans."""
        self._approval_callback = fn

    def set_progress_callback(self, fn: Callable[[PlanStep, str], None]):
        """Set a callback for progress reporting during execution."""
        self._progress_callback = fn

    # ── Main API ───────────────────────────────────────────────────────────

    def plan_and_execute(self, request: str, auto_approve: bool = False) -> Tuple[str, ExecutionPlan]:
        """
        Full pipeline: plan → approve → execute → report.

        If auto_approve is False and no callback is set, returns the plan
        with approval_text for the caller to present.
        """
        plan = self.plan(request)

        if auto_approve:
            self.approve(plan)
            response = self.execute(plan)
            return response, plan

        # If a callback is registered, always route through it
        if self._approval_callback:
            approved = self._approval_callback(plan)
            if approved:
                self.approve(plan)
                response = self.execute(plan)
                return response, plan
            else:
                for step in plan.steps:
                    step.status = StepStatus.SKIPPED
                return "Plan cancelled. No actions were taken.", plan

        # No callback — auto-approve safe plans, prompt for the rest
        if not plan.needs_approval:
            self.approve(plan)
            response = self.execute(plan)
            return response, plan

        # Return plan text for manual approval
        return plan.approval_text(), plan

    def plan(self, request: str) -> ExecutionPlan:
        """Analyze a request and build an execution plan."""

        # 1. Try complex task patterns first
        for pattern, task_type, step_templates in _TASK_PATTERNS:
            m = re.search(pattern, request)
            if m:
                params = m.groupdict()
                return self._build_template_plan(request, task_type, step_templates, params)

        # 2. Try word-cloud intent resolution (handles simple + multi-clause)
        resolved = self.resolver.resolve_multi(request)
        if not resolved:
            single = self.resolver.resolve(request)
            if single:
                resolved = [single]

        if resolved:
            return self._build_intent_plan(request, resolved)

        # 3. Fallback — try to reason about unknown requests
        return self._build_reasoning_plan(request)

    def approve(self, plan: ExecutionPlan):
        """Mark a plan as approved for execution."""
        plan.approved = True
        for step in plan.steps:
            if step.status == StepStatus.PENDING:
                step.status = StepStatus.APPROVED

    def execute(self, plan: ExecutionPlan) -> str:
        """
        Execute an approved plan with parallel processing.

        Steps in the same parallel_group with all dependencies satisfied
        run concurrently via ThreadPoolExecutor.

        Returns an intelligent natural language + ASCII formatted report.
        """
        if not plan.approved:
            return "Plan has not been approved. Please approve before executing."

        step_results: List[Tuple[PlanStep, str]] = []
        group_ids = sorted(plan.parallel_groups.keys())

        for gid in group_ids:
            group_steps = plan.parallel_groups[gid]

            # Filter to only steps whose deps are satisfied
            ready = [s for s in group_steps if self._deps_satisfied(s, plan)]
            if not ready:
                continue

            if len(ready) == 1:
                result_text = self._execute_step(ready[0])
                step_results.append((ready[0], result_text))
            else:
                results = self._execute_parallel(ready)
                for step, text in zip(ready, results):
                    step_results.append((step, text))

        plan.executed = True

        # ── Build rich NL + ASCII output ──────────────────────────
        return self._compose_execution_report(plan, step_results)

    def _compose_execution_report(self, plan: ExecutionPlan,
                                   step_results: List[Tuple["PlanStep", str]]) -> str:
        """Build a rich natural language + ASCII report of execution."""
        parts: List[str] = []

        done = sum(1 for s in plan.steps if s.status == StepStatus.DONE)
        failed = sum(1 for s in plan.steps if s.status == StepStatus.FAILED)
        total = len(plan.steps)

        # Header — intelligent summary
        if total == 1 and done == 1:
            step = plan.steps[0]
            desc = step.description
            parts.append(f"Done! {desc}.")
            if step.result and step.result.output:
                parts.append(f"\n{step.result.output[:200]}")
        elif failed == 0:
            parts.append(f"All {total} steps completed successfully. Here's what I did:\n")
        else:
            parts.append(f"Completed {done}/{total} steps ({failed} had issues):\n")

        # Step-by-step progress (only for multi-step plans)
        if total > 1:
            progress_items = []
            for step in plan.steps:
                status_map = {
                    StepStatus.DONE: "done",
                    StepStatus.FAILED: "failed",
                    StepStatus.RUNNING: "running",
                    StepStatus.SKIPPED: "skipped",
                }
                status = status_map.get(step.status, "pending")
                progress_items.append((step.description, status))

            parts.append(_ascii_progress_block(progress_items))
            parts.append("")

        # File tree — collect all created files/dirs
        created_paths = []
        for step in plan.steps:
            if step.result and step.result.status == ToolStatus.SUCCESS:
                if step.result.path:
                    created_paths.append(step.result.path)
                elif step.action:
                    p = step.action.args.get("path", "")
                    if p:
                        created_paths.append(p)

        if len(created_paths) > 1:
            parts.append(_ascii_file_tree(created_paths))
            parts.append("")

        # Failures detail
        for step in plan.steps:
            if step.status == StepStatus.FAILED and step.result:
                parts.append(f"  ✗ {step.description}: {step.result.message}")

        # Closing
        if total > 1 and failed == 0:
            parts.append(f"Everything looks good! All {total} steps are done.")

        return "\n".join(parts)

    # ── Plan builders ──────────────────────────────────────────────────────

    def _build_template_plan(self, request: str, task_type: str,
                              templates: List[Dict[str, Any]],
                              params: Dict[str, str]) -> ExecutionPlan:
        """Build a plan from a matched task pattern template."""
        steps: List[PlanStep] = []

        for i, tpl in enumerate(templates):
            desc = tpl["desc"].format(**params)
            tool_name = tpl.get("tool", "")
            group = tpl.get("group", i)
            deps = tpl.get("deps", [])
            destructive = tpl.get("destructive", tool_name in _DESTRUCTIVE_TOOLS)

            # Build tool call
            action = None
            if "args_tpl" in tpl:
                args = {k: v.format(**params) for k, v in tpl["args_tpl"].items()}
                action = ToolCall(tool_name, args)
            elif "content_gen" in tpl:
                path, content = _gen_content(tpl["content_gen"], params)
                action = ToolCall("create_file", {"path": path, "content": content})

            steps.append(PlanStep(
                id=i,
                description=desc,
                reasoning=f"Part of {task_type} workflow",
                action=action,
                depends_on=deps,
                parallel_group=group,
                is_destructive=destructive,
            ))

        summary = f"{task_type}: {len(steps)} steps for '{request[:60]}'"
        return ExecutionPlan(request=request, summary=summary, steps=steps)

    def _build_intent_plan(self, request: str, resolved: list) -> ExecutionPlan:
        """Build a plan from word-cloud resolved intents."""
        from synthos.tools.batch import BatchProcessor
        bp = BatchProcessor(self.executor, self.generator)

        steps: List[PlanStep] = []
        # Check which steps can be parallelized
        # Rule: steps that don't create dependencies for each other can be parallel
        dir_steps = set()  # track mkdir steps by name

        for i, score in enumerate(resolved):
            name = score.entities.get("name", "")
            intent = score.intent
            params = {"name": name}
            if score.entities.get("filenames"):
                params["name"] = score.entities["filenames"]
            if score.entities.get("quoted"):
                params["cmd"] = score.entities["quoted"]

            action = bp._intent_to_action(intent, params)

            # Determine dependencies
            deps: List[int] = []
            if action and intent != "mkdir":
                # File creation depends on its parent directory step
                path = params.get("name", "")
                parent = "/".join(path.split("/")[:-1]) if "/" in path else ""
                for ds_id, ds_name in dir_steps:
                    if parent and ds_name in parent:
                        deps.append(ds_id)

            if intent == "mkdir":
                dir_steps.add((i, name))

            # Determine parallel group
            group = 0 if not deps else max(deps) + 1

            destructive = intent in ("run_shell", "delete_file")

            # Generate description
            desc_templates = {
                "mkdir": f"Create directory '{name}'",
                "create_py": f"Write Python file '{name}'",
                "create_sh": f"Write shell script '{name}'",
                "create_md": f"Write Markdown document '{name}'",
                "create_file": f"Create file '{name}'",
                "scaffold": f"Scaffold project '{name}'",
                "list_dir": f"List directory '{name}'",
                "read_file": f"Read file '{name}'",
                "delete_file": f"Delete '{name}'",
                "run_shell": f"Run command",
                "tree": f"Show tree for '{name}'",
                "file_exists": f"Check if '{name}' exists",
            }
            desc = desc_templates.get(intent, f"Execute {intent} for '{name}'")

            steps.append(PlanStep(
                id=i,
                description=desc,
                reasoning=f"Matched intent '{intent}' with confidence {score.confidence:.0%}",
                action=action,
                depends_on=deps,
                parallel_group=group,
                is_destructive=destructive,
                confidence=score.confidence,
            ))

        summary = f"{len(steps)} task(s) from: {request[:60]}"
        return ExecutionPlan(request=request, summary=summary, steps=steps)

    def _build_reasoning_plan(self, request: str) -> ExecutionPlan:
        """Build a plan for unknown requests by decomposing into primitives."""
        steps: List[PlanStep] = []

        # Try to extract any actionable tokens
        tokens = re.findall(r"[\w./-]+", request.lower())

        # Look for file-like tokens
        file_tokens = [t for t in tokens if "." in t and not t.startswith(".")]
        dir_tokens = [t for t in tokens if "/" in t]
        remaining_tokens = [t for t in tokens if t not in file_tokens and t not in dir_tokens]

        step_id = 0

        # Create directories for any path-like tokens
        created_dirs: Set[str] = set()
        for dt in dir_tokens:
            parent = "/".join(dt.split("/")[:-1])
            if parent and parent not in created_dirs:
                steps.append(PlanStep(
                    id=step_id,
                    description=f"Create directory '{parent}'",
                    reasoning="Path-like token detected — ensuring parent directory exists",
                    action=ToolCall("create_directory", {"path": parent}),
                    parallel_group=0,
                ))
                created_dirs.add(parent)
                step_id += 1

        # Create files for file-like tokens
        for ft in file_tokens:
            deps = []
            parent = "/".join(ft.split("/")[:-1]) if "/" in ft else ""
            for s in steps:
                if s.action and s.action.args.get("path", "") == parent:
                    deps.append(s.id)

            content = self._infer_file_content(ft)
            steps.append(PlanStep(
                id=step_id,
                description=f"Create file '{ft}'",
                reasoning="Detected filename pattern — generating appropriate content",
                action=ToolCall("create_file", {"path": ft, "content": content}),
                depends_on=deps,
                parallel_group=1 if deps else 0,
            ))
            step_id += 1

        if not steps:
            # Truly unknown — create a reasoning step
            steps.append(PlanStep(
                id=0,
                description=f"Analyze: {request}",
                reasoning="I'm not sure exactly what system action to take. Let me think about this.",
                confidence=0.4,
            ))

        summary = f"Inferred {len(steps)} step(s) from: {request[:60]}"
        return ExecutionPlan(request=request, summary=summary, steps=steps)

    # ── Execution engine ───────────────────────────────────────────────────

    def _execute_step(self, step: PlanStep) -> str:
        """Execute a single plan step."""
        step.status = StepStatus.RUNNING
        if self._progress_callback:
            self._progress_callback(step, "running")

        if not step.action:
            step.status = StepStatus.DONE
            if self._progress_callback:
                self._progress_callback(step, "done (reasoning only)")
            return f"Step {step.id}: {step.description} — (reasoning step, no system action)"

        # Handle scaffold specially
        if "scaffold" in step.description.lower() and step.action.name == "create_directory":
            name = step.action.args.get("path", "project")
            from synthos.tools.filesystem import FileSystemTools
            fs = FileSystemTools(self.executor)
            structure = {
                "src": {"__init__.py": "", "main.py": f'def main():\n    print("{name} running")\n\nif __name__ == "__main__":\n    main()\n'},
                "tests": {"__init__.py": "", "test_main.py": "def test_ok():\n    assert True\n"},
                "README.md": f"# {name}\n",
            }
            results = fs.scaffold_project(name, structure)
            failed = [r for r in results if r.status == ToolStatus.ERROR]
            if failed:
                step.status = StepStatus.FAILED
                step.result = failed[0]
                return f"Step {step.id}: {step.description} — FAILED: {failed[0].message}"
            step.status = StepStatus.DONE
            step.result = results[0]
            return f"Step {step.id}: {step.description} — Done! Project scaffolded."

        result = self.executor.execute(step.action)
        step.result = result

        if result.status == ToolStatus.SUCCESS:
            step.status = StepStatus.DONE
            if self._progress_callback:
                self._progress_callback(step, "done")
            output_preview = ""
            if result.output and len(result.output) < 200:
                output_preview = f"\n  Output: {result.output}"
            return f"Step {step.id}: {step.description} — Done!{output_preview}"
        else:
            step.status = StepStatus.FAILED
            if self._progress_callback:
                self._progress_callback(step, f"failed: {result.message}")
            return f"Step {step.id}: {step.description} — Failed: {result.message}"

    def _execute_parallel(self, steps: List[PlanStep]) -> List[str]:
        """Execute multiple steps in parallel using threads."""
        results: List[str] = [""] * len(steps)

        with concurrent.futures.ThreadPoolExecutor(max_workers=min(len(steps), 8)) as pool:
            future_to_idx = {}
            for i, step in enumerate(steps):
                future = pool.submit(self._execute_step, step)
                future_to_idx[future] = i

            for future in concurrent.futures.as_completed(future_to_idx):
                idx = future_to_idx[future]
                try:
                    results[idx] = future.result()
                except Exception as exc:
                    steps[idx].status = StepStatus.FAILED
                    results[idx] = f"Step {steps[idx].id}: {steps[idx].description} — Error: {exc}"

        return results

    def _deps_satisfied(self, step: PlanStep, plan: ExecutionPlan) -> bool:
        """Check if all dependencies of a step are satisfied."""
        if not step.depends_on:
            return True
        for dep_id in step.depends_on:
            dep_step = next((s for s in plan.steps if s.id == dep_id), None)
            if dep_step and dep_step.status not in (StepStatus.DONE, StepStatus.SKIPPED):
                return False
        return True

    # ── Helpers ────────────────────────────────────────────────────────────

    def _infer_file_content(self, filename: str) -> str:
        """Infer appropriate file content based on extension."""
        if filename.endswith(".py"):
            modname = filename.replace(".py", "").split("/")[-1].replace("-", "_")
            return f'"""\n{modname} — auto-generated by SYNTHOS.\n"""\n\ndef main():\n    print("{modname} running")\n\nif __name__ == "__main__":\n    main()\n'
        if filename.endswith(".sh"):
            return f"#!/usr/bin/env bash\nset -euo pipefail\necho 'Running {filename}'\n"
        if filename.endswith(".md"):
            title = filename.replace(".md", "").split("/")[-1].replace("-", " ").replace("_", " ").title()
            return f"# {title}\n\nGenerated by SYNTHOS.\n"
        if filename.endswith((".yaml", ".yml")):
            return f"# {filename} — generated by SYNTHOS\n"
        if filename.endswith(".json"):
            return "{}\n"
        if filename.endswith(".toml"):
            return f"# {filename}\n[project]\nname = \"{filename.split('/')[0]}\"\n"
        return f"# {filename}\n# generated by SYNTHOS\n"


# ═══════════════════════════════════════════════════════════════════════════════
# ASCII formatting helpers for execution reports
# ═══════════════════════════════════════════════════════════════════════════════

def _ascii_progress_block(items: List[Tuple[str, str]]) -> str:
    """Render a vertical step-by-step progress indicator."""
    lines = []
    for i, (label, status) in enumerate(items):
        if status == "done":
            icon = "✓"
        elif status == "failed":
            icon = "✗"
        elif status == "running":
            icon = "►"
        elif status == "skipped":
            icon = "○"
        else:
            icon = "·"
        connector = "│" if i < len(items) - 1 else " "
        lines.append(f"  {icon} {label}")
        if i < len(items) - 1:
            lines.append(f"  {connector}")
    return "\n".join(lines)


def _ascii_file_tree(paths: List[str]) -> str:
    """Build an ASCII file tree from a list of absolute or relative paths."""
    if not paths:
        return ""

    # Find the longest common prefix to strip absolute paths
    normalized = [p.replace("\\", "/").rstrip("/") for p in paths]
    if len(normalized) > 1:
        common = os.path.commonpath(normalized)
        # Strip the common prefix, keep the project-level paths
        clean = [p[len(common):].lstrip("/") for p in normalized]
    else:
        # Single path — just take the last 2 components
        parts = normalized[0].split("/")
        clean = ["/".join(parts[-2:]) if len(parts) > 2 else normalized[0]]

    # Build tree structure
    tree: Dict[str, Any] = {}
    for path in clean:
        parts = path.split("/")
        current = tree
        for i, part in enumerate(parts):
            if not part:
                continue
            if i == len(parts) - 1 and "." in part:
                current[part] = ""  # file
            else:
                current = current.setdefault(part, {})

    # Render
    lines = ["Created:"]
    _render_tree(tree, lines, "  ")
    return "\n".join(lines)


def _render_tree(tree: Dict[str, Any], lines: List[str], prefix: str) -> None:
    items = list(tree.items())
    for i, (name, children) in enumerate(items):
        is_last = (i == len(items) - 1)
        connector = "└── " if is_last else "├── "
        child_prefix = prefix + ("    " if is_last else "│   ")

        if isinstance(children, dict) and children:
            lines.append(f"{prefix}{connector}📁 {name}/")
            _render_tree(children, lines, child_prefix)
        elif isinstance(children, dict):
            lines.append(f"{prefix}{connector}📁 {name}/")
        else:
            ext = name.rsplit(".", 1)[-1] if "." in name else ""
            icon = {"py": "🐍", "sh": "⚙️", "md": "📝", "txt": "📄"}.get(ext, "📄")
            lines.append(f"{prefix}{connector}{icon} {name}")
