"""
File Generators — template-based .md, .sh, .py generation with LM distillation.

Each generator uses structured templates seeded by intent decomposition,
then applies chain-of-thought distillation to produce coherent, well-
documented output files.
"""

from __future__ import annotations

import re
import textwrap
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional

from synthos.tools.executor import ToolExecutor, ToolCall, ToolResult


# ═══════════════════════════════════════════════════════════════════════════════
# Template library
# ═══════════════════════════════════════════════════════════════════════════════

_PY_MODULE = '''\
"""
{title}
{underline}

{description}

Generated by SYNTHOS SynthoLM · {timestamp}
"""

{imports}

{body}


if __name__ == "__main__":
    main()
'''

_PY_CLASS = '''\
class {class_name}:
    """{docstring}"""

    def __init__(self{init_params}):
{init_body}

{methods}
'''

_PY_FUNCTION = '''\
def {name}({params}){ret_annotation}:
    """{docstring}"""
{body}
'''

_SH_SCRIPT = '''\
#!/usr/bin/env bash
# ─────────────────────────────────────────────────────────────────────
# {title}
# {description}
# Generated by SYNTHOS SynthoLM · {timestamp}
# ─────────────────────────────────────────────────────────────────────
set -euo pipefail

{body}
'''

_MD_DOC = '''\
# {title}

> {description}

*Generated by SYNTHOS SynthoLM · {timestamp}*

---

{body}
'''

_MD_README = '''\
# {project_name}

{description}

## Installation

```bash
{install_steps}
```

## Usage

```bash
{usage_steps}
```

## Structure

```
{structure}
```

## License

{license}
'''


class FileGenerator:
    """
    Generate .py, .sh, .md files from high-level descriptions.

    Uses template distillation: a description is decomposed into
    structural elements, filled into templates, and post-processed
    for coherence.
    """

    def __init__(self, executor: ToolExecutor):
        self.executor = executor
        self._ts = lambda: datetime.now().strftime("%Y-%m-%d %H:%M")

    # ── Python ─────────────────────────────────────────────────────────────

    def generate_python_module(
        self,
        path: str,
        title: str,
        description: str = "",
        imports: Optional[List[str]] = None,
        functions: Optional[List[Dict[str, str]]] = None,
        classes: Optional[List[Dict[str, Any]]] = None,
        main_body: str = "    pass",
    ) -> ToolResult:
        """Generate a complete Python module."""
        imp_block = "\n".join(imports or [])

        parts: List[str] = []

        # Classes
        for cls in (classes or []):
            methods_code = ""
            for m in cls.get("methods", []):
                methods_code += _PY_FUNCTION.format(
                    name=m.get("name", "method"),
                    params="self" + (", " + m["params"] if m.get("params") else ""),
                    ret_annotation=f" -> {m['returns']}" if m.get("returns") else "",
                    docstring=m.get("docstring", ""),
                    body=textwrap.indent(m.get("body", "pass"), "    "),
                ) + "\n"

            init_params_str = ""
            init_body_str = "        pass"
            if cls.get("init_params"):
                init_params_str = ", " + ", ".join(cls["init_params"])
                assigns = []
                for p in cls["init_params"]:
                    name = p.split(":")[0].split("=")[0].strip()
                    assigns.append(f"        self.{name} = {name}")
                init_body_str = "\n".join(assigns)

            parts.append(_PY_CLASS.format(
                class_name=cls.get("name", "MyClass"),
                docstring=cls.get("docstring", ""),
                init_params=init_params_str,
                init_body=init_body_str,
                methods=textwrap.indent(methods_code, "    "),
            ))

        # Functions
        for fn in (functions or []):
            parts.append(_PY_FUNCTION.format(
                name=fn.get("name", "my_function"),
                params=fn.get("params", ""),
                ret_annotation=f" -> {fn['returns']}" if fn.get("returns") else "",
                docstring=fn.get("docstring", ""),
                body=textwrap.indent(fn.get("body", "pass"), "    "),
            ))

        # Main
        parts.append(f"def main():\n{textwrap.indent(main_body, '    ')}")

        body = "\n\n".join(parts)
        underline = "=" * len(title)

        content = _PY_MODULE.format(
            title=title, underline=underline, description=description,
            timestamp=self._ts(), imports=imp_block, body=body,
        )

        return self.executor.execute(ToolCall("create_file", {"path": path, "content": content}))

    def generate_python_script(self, path: str, description: str, code: str) -> ToolResult:
        """Generate a simple Python script with header."""
        header = f'"""\n{description}\nGenerated by SYNTHOS SynthoLM · {self._ts()}\n"""\n\n'
        return self.executor.execute(ToolCall("create_file", {"path": path, "content": header + code}))

    # ── Shell ──────────────────────────────────────────────────────────────

    def generate_shell_script(
        self,
        path: str,
        title: str,
        description: str = "",
        body: str = 'echo "SYNTHOS shell script"',
    ) -> ToolResult:
        content = _SH_SCRIPT.format(
            title=title, description=description,
            timestamp=self._ts(), body=body,
        )
        result = self.executor.execute(ToolCall("create_file", {"path": path, "content": content}))
        # Make executable
        if result.status.value == "success":
            p = Path(result.path) if result.path else self.executor._resolve(path)
            try:
                p.chmod(0o755)
            except OSError:
                pass
        return result

    # ── Markdown ───────────────────────────────────────────────────────────

    def generate_markdown(
        self,
        path: str,
        title: str,
        description: str = "",
        body: str = "",
    ) -> ToolResult:
        content = _MD_DOC.format(
            title=title, description=description,
            timestamp=self._ts(), body=body,
        )
        return self.executor.execute(ToolCall("create_file", {"path": path, "content": content}))

    def generate_readme(
        self,
        path: str,
        project_name: str,
        description: str = "",
        install_steps: str = "pip install .",
        usage_steps: str = "python main.py",
        structure: str = ".",
        license_text: str = "MIT",
    ) -> ToolResult:
        content = _MD_README.format(
            project_name=project_name, description=description,
            install_steps=install_steps, usage_steps=usage_steps,
            structure=structure, license=license_text,
        )
        return self.executor.execute(ToolCall("create_file", {"path": path, "content": content}))

    # ── Batch generation ───────────────────────────────────────────────────

    def generate_project(
        self,
        name: str,
        description: str,
        modules: Optional[List[Dict[str, Any]]] = None,
        scripts: Optional[List[Dict[str, str]]] = None,
        docs: Optional[List[Dict[str, str]]] = None,
    ) -> List[ToolResult]:
        """Generate a complete project with .py modules, .sh scripts, .md docs."""
        results: List[ToolResult] = []

        # Root directory
        results.append(self.executor.execute(ToolCall("create_directory", {"path": name})))

        # Python modules
        for mod in (modules or []):
            r = self.generate_python_module(
                path=f"{name}/{mod['path']}",
                title=mod.get("title", mod["path"]),
                description=mod.get("description", ""),
                imports=mod.get("imports"),
                functions=mod.get("functions"),
                classes=mod.get("classes"),
                main_body=mod.get("main_body", "pass"),
            )
            results.append(r)

        # Shell scripts
        for sc in (scripts or []):
            r = self.generate_shell_script(
                path=f"{name}/{sc['path']}",
                title=sc.get("title", sc["path"]),
                description=sc.get("description", ""),
                body=sc.get("body", "echo done"),
            )
            results.append(r)

        # Markdown docs
        for doc in (docs or []):
            r = self.generate_markdown(
                path=f"{name}/{doc['path']}",
                title=doc.get("title", doc["path"]),
                description=doc.get("description", ""),
                body=doc.get("body", ""),
            )
            results.append(r)

        # Auto-generate README
        results.append(self.generate_readme(
            path=f"{name}/README.md",
            project_name=name,
            description=description,
        ))

        return results
