"""
FileSystem Tools — high-level file operations for SYNTHOS.

Wraps the ToolExecutor primitives into semantic operations like
"create a project structure" or "scaffold a Python package".
"""

from __future__ import annotations

from pathlib import Path
from typing import Dict, List, Optional, Tuple

from synthos.tools.executor import ToolExecutor, ToolCall, ToolResult, ToolStatus


class FileSystemTools:
    """
    High-level filesystem operations built on ToolExecutor.

    Provides compound actions like scaffolding project trees, creating
    file batches, and reading directory contents for reasoning.
    """

    def __init__(self, executor: ToolExecutor):
        self.executor = executor

    # ── Compound operations ────────────────────────────────────────────────

    def scaffold_project(self, name: str, structure: Dict) -> List[ToolResult]:
        """
        Create a directory tree from a nested dict.

        Example::

            structure = {
                "src": {
                    "__init__.py": "",
                    "main.py": "print('hello')",
                },
                "tests": {
                    "test_main.py": "...",
                },
                "README.md": "# My Project",
            }
        """
        calls = self._flatten_structure(name, structure)
        return self.executor.execute_batch(calls)

    def create_files_batch(self, files: Dict[str, str]) -> List[ToolResult]:
        """Create multiple files at once. Keys are paths, values are content."""
        calls = []
        for path, content in files.items():
            calls.append(ToolCall("create_file", {"path": path, "content": content}))
        return self.executor.execute_batch(calls)

    def read_directory_context(self, path: str = ".", depth: int = 2) -> str:
        """Read a directory tree and summarise for the reasoning engine."""
        tree_result = self.executor.execute(ToolCall("tree", {"path": path, "depth": depth}))
        if tree_result.status != ToolStatus.SUCCESS:
            return f"Could not read directory: {tree_result.message}"
        return tree_result.output

    def ensure_directory(self, path: str) -> ToolResult:
        return self.executor.execute(ToolCall("create_directory", {"path": path}))

    def write_file(self, path: str, content: str) -> ToolResult:
        return self.executor.execute(ToolCall("create_file", {"path": path, "content": content}))

    def read_file(self, path: str) -> str:
        result = self.executor.execute(ToolCall("read_file", {"path": path}))
        return result.output if result.status == ToolStatus.SUCCESS else ""

    def exists(self, path: str) -> bool:
        result = self.executor.execute(ToolCall("file_exists", {"path": path}))
        import json
        try:
            return json.loads(result.output).get("exists", False)
        except Exception:
            return False

    # ── Internal ───────────────────────────────────────────────────────────

    def _flatten_structure(self, base: str, tree: Dict, calls: Optional[List[ToolCall]] = None) -> List[ToolCall]:
        if calls is None:
            calls = [ToolCall("create_directory", {"path": base})]
        for name, value in tree.items():
            child = f"{base}/{name}"
            if isinstance(value, dict):
                calls.append(ToolCall("create_directory", {"path": child}))
                self._flatten_structure(child, value, calls)
            else:
                calls.append(ToolCall("create_file", {"path": child, "content": str(value)}))
        return calls
