"""
Batch Processor — multi-task execution with LM distillation techniques.

Implements:
  1. Word-cloud fuzzy matching — order-independent intent resolution
  2. Chain-of-Thought (CoT) decomposition — break complex requests into steps
  3. Self-consistency — validate outputs against intent
  4. Knowledge distillation — compress reasoning into structured templates
  5. Task batching — execute many file/dir operations in sequence
  6. Natural language output — always respond in readable English
"""

from __future__ import annotations

import re
import json
from dataclasses import dataclass, field
from typing import Any, Dict, List, Optional, Tuple

from synthos.tools.executor import ToolExecutor, ToolCall, ToolResult, ToolStatus
from synthos.tools.generators import FileGenerator


@dataclass
class ReasoningStep:
    """One step in a chain-of-thought decomposition."""
    step_num: int
    thought: str
    action: Optional[ToolCall] = None
    result: Optional[ToolResult] = None
    confidence: float = 1.0


@dataclass
class TaskPlan:
    """A decomposed plan for a complex request."""
    original_request: str
    summary: str
    steps: List[ReasoningStep] = field(default_factory=list)
    word_cloud: Optional[Dict[str, Any]] = None
    completed: bool = False

    def to_dict(self) -> Dict[str, Any]:
        return {
            "request": self.original_request,
            "summary": self.summary,
            "steps": [
                {
                    "step": s.step_num,
                    "thought": s.thought,
                    "action": repr(s.action) if s.action else None,
                    "status": s.result.status.value if s.result else "pending",
                    "confidence": s.confidence,
                }
                for s in self.steps
            ],
            "word_cloud": self.word_cloud,
            "completed": self.completed,
        }


# ── Distillation templates for natural-language thoughts ──────────────────────

_DISTILL_TEMPLATES: Dict[str, str] = {
    "mkdir": "I need to create a directory called '{name}'. Let me verify the path is safe and create it.",
    "create_py": "I need to create a Python file '{name}'. I'll generate a well-structured module with docstring, imports, and a main block.",
    "create_sh": "I need to create a shell script '{name}'. I'll add a shebang, safety flags (set -euo pipefail), and the requested functionality.",
    "create_md": "I need to create a markdown document '{name}'. I'll structure it with a title, description, and organized sections.",
    "create_file": "I need to create a file called '{name}'.",
    "scaffold": "I need to scaffold a project called '{name}'. I'll create the directory structure, __init__.py files, a README, and any requested modules.",
    "list_dir": "I'll list the contents of the directory '{name}'.",
    "read_file": "I'll read the file '{name}' and show its contents.",
    "delete_file": "I'll delete the file '{name}'.",
    "run_shell": "I'll run the shell command: {name}",
    "tree": "I'll show the directory tree for '{name}'.",
    "file_exists": "I'll check whether '{name}' exists.",
    "append": "I'll append content to '{name}'.",
}

# ── Natural language response templates ───────────────────────────────────────

_NL_RESPONSES: Dict[str, Dict[str, str]] = {
    "mkdir": {
        "success": "Done! I created the directory '{name}' for you.",
        "error": "I wasn't able to create the directory '{name}': {reason}",
    },
    "create_py": {
        "success": "Done! I wrote a Python file at '{path}'. It includes a module docstring, imports, and a main() entry point.",
        "error": "I had trouble creating the Python file '{name}': {reason}",
    },
    "create_sh": {
        "success": "Done! I wrote a shell script at '{path}' with a shebang and safety flags.",
        "error": "I couldn't create the shell script '{name}': {reason}",
    },
    "create_md": {
        "success": "Done! I wrote a Markdown document at '{path}' with a title and section structure.",
        "error": "I couldn't create the Markdown file '{name}': {reason}",
    },
    "create_file": {
        "success": "Done! I created the file '{path}'.",
        "error": "I couldn't create the file '{name}': {reason}",
    },
    "scaffold": {
        "success": "Done! I scaffolded the project '{name}' with src/, tests/, scripts/, and docs/ directories, along with starter files and a README.",
        "error": "I had trouble scaffolding the project '{name}': {reason}",
    },
    "list_dir": {
        "success": "Here are the contents of '{name}':\n\n{output}",
        "error": "I couldn't list the directory '{name}': {reason}",
    },
    "read_file": {
        "success": "Here are the contents of '{name}':\n\n{output}",
        "error": "I couldn't read '{name}': {reason}",
    },
    "delete_file": {
        "success": "Done! I deleted '{name}'.",
        "error": "I couldn't delete '{name}': {reason}",
    },
    "run_shell": {
        "success": "I ran the command and here's the output:\n\n{output}",
        "error": "The command failed: {reason}",
    },
    "tree": {
        "success": "Here's the directory tree:\n\n{output}",
        "error": "I couldn't generate the tree: {reason}",
    },
    "file_exists": {
        "success": "{output}",
        "error": "I couldn't check: {reason}",
    },
}


class BatchProcessor:
    """
    Processes natural language requests by:
    1. Word-cloud scoring for order-independent intent matching
    2. Entity extraction from any position in the sentence
    3. Chain-of-thought decomposition into reasoning steps
    4. Tool execution for each step
    5. Natural language response generation
    """

    def __init__(self, executor: ToolExecutor, generator: FileGenerator):
        self.executor = executor
        self.generator = generator
        self._plans: List[TaskPlan] = []
        # Lazy import to avoid circular dependency
        from synthos.lm.wordcloud import IntentResolver
        self.resolver = IntentResolver()

    @property
    def history(self) -> List[TaskPlan]:
        return list(self._plans)

    # ── Public API ─────────────────────────────────────────────────────────

    def process(self, request: str) -> Tuple[str, TaskPlan]:
        """
        Process a natural language request. Returns (response_text, plan).

        This is the main entry point. It:
        1. Parses the request into intents
        2. Builds a chain-of-thought plan
        3. Executes the plan
        4. Validates and reports
        """
        intents = self._parse_intents(request)

        if not intents:
            # No actionable intent — treat as a reasoning/explanation task
            plan = TaskPlan(
                original_request=request,
                summary="Reasoning about: " + request[:80],
                steps=[ReasoningStep(1, f"Understanding request: {request}", confidence=0.8)],
            )
            response = self._reason_about(request)
            plan.completed = True
            self._plans.append(plan)
            return response, plan

        plan = self._build_plan(request, intents)
        response = self._execute_plan(plan)
        plan.completed = True
        self._plans.append(plan)
        return response, plan

    def process_batch(self, requests: List[str]) -> List[Tuple[str, TaskPlan]]:
        """Process multiple requests sequentially."""
        return [self.process(req) for req in requests]

    # ── Intent parsing (word-cloud based, order-independent) ───────────────

    def _parse_intents(self, request: str) -> List[Tuple[str, Dict[str, str]]]:
        """Extract actionable intents using word-cloud fuzzy matching.

        This is ORDER-INDEPENDENT — words can appear in any position.
        The word cloud scores every token against all possible intents
        and the highest-scoring intent wins.
        """
        resolved = self.resolver.resolve_multi(request)

        if not resolved:
            # Try the whole request as a single clause
            single = self.resolver.resolve(request)
            if single:
                resolved = [single]

        intents = []
        for score in resolved:
            params = {"name": score.entities.get("name", "")}
            if score.entities.get("filenames"):
                params["name"] = score.entities["filenames"]
            if score.entities.get("quoted"):
                params["cmd"] = score.entities["quoted"]
            intents.append((score.intent, params))

        return intents

    # ── Plan building (CoT decomposition) ──────────────────────────────────

    def _build_plan(self, request: str, intents: List[Tuple[str, Dict[str, str]]]) -> TaskPlan:
        """Build a chain-of-thought plan from parsed intents."""
        steps: List[ReasoningStep] = []

        for i, (intent_type, params) in enumerate(intents, 1):
            # Distillation: generate a thought from template
            template = _DISTILL_TEMPLATES.get(intent_type, "Processing: {name}")
            fmt_args = {"name": params.get("name", ""), "topic": params.get("topic", "")}
            fmt_args.update({k: v for k, v in params.items() if v and k not in fmt_args})
            thought = template.format_map(fmt_args)

            # Map intent → tool call
            action = self._intent_to_action(intent_type, params)

            steps.append(ReasoningStep(
                step_num=i,
                thought=thought,
                action=action,
                confidence=0.9 if action else 0.6,
            ))

        # Generate word cloud summary for the plan
        cloud = self.resolver.get_word_cloud_summary(request)

        summary = f"{len(steps)} action(s) planned from: {request[:60]}"
        return TaskPlan(original_request=request, summary=summary, steps=steps, word_cloud=cloud)

    def _intent_to_action(self, intent_type: str, params: Dict[str, str]) -> Optional[ToolCall]:
        """Convert a parsed intent into a concrete ToolCall."""
        name = params.get("name", "").strip("'\"")
        path = params.get("path", "").strip("'\"") or "."
        cmd = params.get("cmd", "").strip("'\"")

        if intent_type == "mkdir":
            return ToolCall("create_directory", {"path": name})

        elif intent_type == "create_py":
            if not name.endswith(".py"):
                name += ".py"
            return ToolCall("create_file", {"path": name, "content": self._gen_py_content(name)})

        elif intent_type == "create_sh":
            if not name.endswith(".sh"):
                name += ".sh"
            return ToolCall("create_file", {"path": name, "content": self._gen_sh_content(name)})

        elif intent_type == "create_md":
            if not name.endswith(".md"):
                name += ".md"
            return ToolCall("create_file", {"path": name, "content": self._gen_md_content(name)})

        elif intent_type == "create_file":
            return ToolCall("create_file", {"path": name, "content": ""})

        elif intent_type == "list_dir":
            return ToolCall("list_directory", {"path": path})

        elif intent_type == "read_file":
            return ToolCall("read_file", {"path": path})

        elif intent_type == "delete_file":
            return ToolCall("delete_file", {"path": path})

        elif intent_type == "run_shell":
            return ToolCall("run_shell", {"command": cmd})

        elif intent_type == "scaffold":
            return ToolCall("create_directory", {"path": name})
            # Note: scaffold is handled specially in _execute_plan

        elif intent_type == "help":
            return None  # handled as reasoning

        return None

    # ── Plan execution (natural language output) ──────────────────────────

    def _execute_plan(self, plan: TaskPlan) -> str:
        """Execute all steps and return a natural language response."""
        nl_parts: List[str] = []

        for step in plan.steps:
            intent = self._step_intent(step)
            name = step.action.args.get("path", "") if step.action else ""

            if step.action:
                # Check if this is a scaffold intent
                if intent == "scaffold":
                    results = self._execute_scaffold(name or "project")
                    step.result = results[0] if results else ToolResult(
                        tool="scaffold", status=ToolStatus.SUCCESS, message="Scaffolded project"
                    )
                    failed = [r for r in results if r.status == ToolStatus.ERROR]
                    if not failed:
                        nl_parts.append(self._nl_response("scaffold", ToolStatus.SUCCESS, name=name))
                    else:
                        nl_parts.append(self._nl_response("scaffold", ToolStatus.ERROR, name=name, reason=failed[0].message))
                else:
                    step.result = self.executor.execute(step.action)
                    nl_parts.append(self._nl_response(
                        intent, step.result.status,
                        name=name, path=step.result.path or name,
                        output=step.result.output or "",
                        reason=step.result.message,
                    ))
            else:
                nl_parts.append(step.thought)

        return "\n\n".join(nl_parts)

    def _step_intent(self, step: ReasoningStep) -> str:
        """Infer the intent type from a step's thought and action."""
        thought_lower = step.thought.lower()
        if "scaffold" in thought_lower:
            return "scaffold"
        if step.action:
            tool = step.action.name
            # Map tool names back to intent types
            tool_to_intent = {
                "create_directory": "mkdir",
                "create_file": self._infer_file_intent(step),
                "list_directory": "list_dir",
                "read_file": "read_file",
                "delete_file": "delete_file",
                "run_shell": "run_shell",
                "run_python": "run_shell",
                "tree": "tree",
                "file_exists": "file_exists",
                "append_file": "append",
            }
            return tool_to_intent.get(tool, "create_file")
        return "create_file"

    def _infer_file_intent(self, step: ReasoningStep) -> str:
        """Infer file type from filename."""
        path = step.action.args.get("path", "") if step.action else ""
        if path.endswith(".py"):
            return "create_py"
        if path.endswith(".sh"):
            return "create_sh"
        if path.endswith(".md"):
            return "create_md"
        return "create_file"

    def _nl_response(self, intent: str, status: ToolStatus, **kwargs) -> str:
        """Generate a natural language response from intent + result."""
        templates = _NL_RESPONSES.get(intent, {
            "success": "Done! Task completed for '{name}'.",
            "error": "Something went wrong with '{name}': {reason}",
        })
        key = "success" if status == ToolStatus.SUCCESS else "error"
        template = templates[key]
        # Fill all possible placeholders, defaulting missing ones to empty
        fmt = {"name": "", "path": "", "output": "", "reason": ""}
        fmt.update(kwargs)
        try:
            return template.format_map(fmt)
        except (KeyError, ValueError):
            return template

    def _execute_scaffold(self, name: str) -> List[ToolResult]:
        """Scaffold a full project structure."""
        from synthos.tools.filesystem import FileSystemTools
        fs = FileSystemTools(self.executor)

        structure = {
            "src": {
                "__init__.py": f'"""{name} — auto-generated by SYNTHOS."""\n',
                "main.py": f'"""{name} main module."""\n\n\ndef main():\n    print("{name} is running")\n\n\nif __name__ == "__main__":\n    main()\n',
            },
            "tests": {
                "__init__.py": "",
                "test_main.py": f'"""Tests for {name}."""\n\nimport pytest\n\n\ndef test_placeholder():\n    assert True\n',
            },
            "scripts": {
                "setup.sh": f'#!/usr/bin/env bash\nset -euo pipefail\necho "Setting up {name}..."\npip install -e .\n',
                "run.sh": f'#!/usr/bin/env bash\nset -euo pipefail\npython -m src.main\n',
            },
            "docs": {
                "README.md": f'# {name}\n\nAuto-generated project.\n\n## Getting Started\n\n```bash\npip install -e .\npython -m src.main\n```\n',
                "ARCHITECTURE.md": f'# {name} Architecture\n\n## Overview\n\nDescribe the architecture here.\n',
            },
        }
        return fs.scaffold_project(name, structure)

    # ── Content generators (distillation) ──────────────────────────────────

    def _gen_py_content(self, filename: str) -> str:
        modname = filename.replace(".py", "").replace("-", "_")
        return f'''"""
{modname} — auto-generated by SYNTHOS SynthoLM.

This module provides a starting point. Edit to add your logic.
"""

from __future__ import annotations
from typing import Any, Dict, List, Optional


def main():
    """Entry point for {modname}."""
    print(f"{modname} is running")


if __name__ == "__main__":
    main()
'''

    def _gen_sh_content(self, filename: str) -> str:
        return f'''#!/usr/bin/env bash
# ─────────────────────────────────────────────────────────
# {filename} — auto-generated by SYNTHOS SynthoLM
# ─────────────────────────────────────────────────────────
set -euo pipefail

echo "{filename} is running"
'''

    def _gen_md_content(self, filename: str) -> str:
        title = filename.replace(".md", "").replace("-", " ").replace("_", " ").title()
        return f'''# {title}

> Auto-generated by SYNTHOS SynthoLM

## Overview

Describe the content here.

## Details

Add details below.
'''

    # ── Reasoning (no-action responses) ────────────────────────────────────

    def _reason_about(self, request: str) -> str:
        """Generate a reasoning response for non-actionable requests."""
        # Check for help intent
        if re.search(r"(?i)(?:help|what\s+can|capabilities|tools)", request):
            return self._help_text()

        return (
            f"I understand: \"{request}\"\n\n"
            f"I can help with system tasks like:\n"
            f"  • Creating directories, .py, .sh, .md files\n"
            f"  • Scaffolding project structures\n"
            f"  • Running shell commands\n"
            f"  • Reading and listing files\n"
            f"  • Batch processing multiple tasks\n\n"
            f"Try asking me to do something specific, like:\n"
            f"  \"Create a python file called processor.py\"\n"
            f"  \"Scaffold a project called my-app\"\n"
            f"  \"Create a directory called models and then write a shell script called train.sh\""
        )

    def _help_text(self) -> str:
        tools = self.executor.available_tools
        return (
            "## SYNTHOS Assistant — Capabilities\n\n"
            "I can understand plain English and perform system tasks:\n\n"
            "**File Operations:**\n"
            "  • \"Create a directory called models\"\n"
            "  • \"Write a python file called train.py\"\n"
            "  • \"Create a shell script called deploy.sh\"\n"
            "  • \"Write a markdown doc called ARCHITECTURE.md\"\n"
            "  • \"Read the file config.yaml\"\n"
            "  • \"List what's in the src directory\"\n\n"
            "**Project Management:**\n"
            "  • \"Scaffold a project called my-ml-pipeline\"\n"
            "  • \"Create a batch of python modules for data processing\"\n\n"
            "**Multi-step Tasks:**\n"
            "  • \"Create a directory called lib, then write a python file called lib/utils.py\"\n"
            "  • \"Make a docs folder and create an ARCHITECTURE.md and API.md inside it\"\n\n"
            "**Reasoning:**\n"
            "  • \"Explain how the SYNTHOS pipeline works\"\n"
            "  • \"What is chain-of-thought reasoning?\"\n\n"
            f"**Available tools:** {', '.join(tools)}\n"
        )
