"""
SYNTHOS FastAPI Backend
========================

Serves the SynthoLM engine and SYNTHOS assistant over HTTP for the
web landing page and any external integrations.

Endpoints:
    POST /api/chat        — Send a message, get a pipeline-processed response
    POST /api/encrypt     — Encrypt text with STC
    POST /api/decrypt     — Decrypt STC envelope
    GET  /api/status      — Pipeline status
    GET  /api/health      — Health check

Start with:
    synthos serve          (via CLI)
    python -m synthos.server   (direct)
    uvicorn synthos.server:app --port 8100
"""

from __future__ import annotations

import uvicorn
from fastapi import FastAPI, HTTPException
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
from typing import Optional

from synthos.lm.engine import SynthoLM, GenerationConfig, ModelMode
from synthos.crypto.stc import SymbolicTopologyCipher, STCKey, STCMode
from synthos.utils.config import load_config
from synthos.utils.log import VerboseLevel

# ── App setup ──────────────────────────────────────────────────────────────────

app = FastAPI(
    title="SYNTHOS API",
    description="Syntax-driven AI assistant with 7-layer pipeline, STC cipher, and model eval.",
    version="3.0.0",
)

app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_methods=["*"],
    allow_headers=["*"],
)

# ── Lazy globals (initialised on first request) ───────────────────────────────

_lm: Optional[SynthoLM] = None
_stc_key: Optional[STCKey] = None


def _get_lm() -> SynthoLM:
    global _lm
    if _lm is None:
        cfg = load_config()
        _lm = SynthoLM(config=cfg, verbose=VerboseLevel.NORMAL)
    return _lm


def _get_key() -> STCKey:
    global _stc_key
    if _stc_key is None:
        _stc_key = STCKey.generate()
    return _stc_key


# ── Request / response models ─────────────────────────────────────────────────

class ChatRequest(BaseModel):
    message: str
    mode: str = "pure_syntax"       # pure_syntax | hybrid
    max_tokens: int = 512
    verbose: bool = False

class ChatResponse(BaseModel):
    response: str
    mode: str
    tokens: int
    latency_ms: float
    model_used: Optional[str] = None

class EncryptRequest(BaseModel):
    text: str
    mode: str = "cbc"

class EncryptResponse(BaseModel):
    ciphertext: str
    key: str
    mode: str

class DecryptRequest(BaseModel):
    ciphertext_hex: str
    key_hex: str
    mode: str = "cbc"

class DecryptResponse(BaseModel):
    plaintext: str

class StatusResponse(BaseModel):
    primitives: int
    lattice_cells: int
    concepts: int
    attention_heads: int
    grammar_rules: int
    coherence: float
    turns: int


# ── Endpoints ──────────────────────────────────────────────────────────────────

@app.get("/api/health")
def health():
    return {"status": "ok", "engine": "SynthoLM", "version": "3.0.0"}


@app.post("/api/chat", response_model=ChatResponse)
def chat(req: ChatRequest):
    lm = _get_lm()
    cfg = GenerationConfig(
        mode=ModelMode.HYBRID if req.mode == "hybrid" else ModelMode.PURE_SYNTAX,
        max_tokens=req.max_tokens,
        verbose=req.verbose,
    )
    result = lm.generate(req.message, cfg)
    return ChatResponse(
        response=result.text,
        mode=result.mode,
        tokens=result.tokens_generated,
        latency_ms=round(result.latency_ms, 1),
        model_used=result.model_used,
    )


@app.post("/api/encrypt", response_model=EncryptResponse)
def encrypt(req: EncryptRequest):
    key = _get_key()
    mode = STCMode(req.mode)
    cipher = SymbolicTopologyCipher(key, mode)
    ct = cipher.encrypt(req.text.encode())
    return EncryptResponse(
        ciphertext=ct.hex(),
        key=key.hex(),
        mode=mode.value,
    )


@app.post("/api/decrypt", response_model=DecryptResponse)
def decrypt(req: DecryptRequest):
    try:
        key = STCKey.from_hex(req.key_hex)
    except Exception:
        raise HTTPException(400, "Invalid key hex")
    try:
        ct = bytes.fromhex(req.ciphertext_hex.replace("\n", "").replace(" ", ""))
    except Exception:
        raise HTTPException(400, "Invalid ciphertext hex")

    mode = STCMode(req.mode)
    cipher = SymbolicTopologyCipher(key, mode)
    try:
        pt = cipher.decrypt(ct)
    except ValueError as exc:
        raise HTTPException(400, str(exc))
    return DecryptResponse(plaintext=pt.decode(errors="replace"))


@app.get("/api/status", response_model=StatusResponse)
def status():
    lm = _get_lm()
    return StatusResponse(
        primitives=len(lm.lpe.primitives),
        lattice_cells=len(lm.gpl.cells),
        concepts=len(lm.scm.concepts),
        attention_heads=len(lm.tam.attention_heads),
        grammar_rules=len(lm.rge.production_rules),
        coherence=lm.scf.compute_coherence(),
        turns=lm.turn_count,
    )


# ── Tool endpoints ─────────────────────────────────────────────────────────────

class TaskRequest(BaseModel):
    task: str
    workspace: Optional[str] = None

class TaskResponse(BaseModel):
    response: str
    mode: str
    steps: int
    latency_ms: float

@app.post("/api/do", response_model=TaskResponse)
def do_task(req: TaskRequest):
    """Execute a plain-English task via the tool system."""
    lm = _get_lm()
    # Override workspace if provided
    if req.workspace:
        from pathlib import Path
        lm.tool_executor.workspace = Path(req.workspace)
        lm.tool_executor.workspace.mkdir(parents=True, exist_ok=True)
    result = lm.generate(req.task)
    plan_steps = 0
    if isinstance(result.pipeline_trace, dict):
        plan = result.pipeline_trace.get("plan", {})
        if isinstance(plan, dict):
            plan_steps = len(plan.get("steps", []))
    return TaskResponse(
        response=result.text,
        mode=result.mode,
        steps=plan_steps,
        latency_ms=round(result.latency_ms, 1),
    )

@app.get("/api/tools")
def list_tools():
    """List available tools."""
    lm = _get_lm()
    return {"tools": lm.tool_executor.available_tools}


# ── Direct launch ──────────────────────────────────────────────────────────────

def run_server(host: str = "0.0.0.0", port: int = 8100):
    """Launch the SYNTHOS API server."""
    uvicorn.run(app, host=host, port=port, log_level="info")


if __name__ == "__main__":
    run_server()
