"""
SynthoLM — Syntax-Based Language Model
========================================

A language model whose "forward pass" is the 7-layer SYNTHOS pipeline.
It can operate in two modes:

1. **Pure-syntax mode** (default) — the pipeline itself generates structured
   responses using pattern matching, grammar rules, substitution chains, and
   template rendering.  No external model needed.

2. **Hybrid mode** — the pipeline pre-processes the prompt (concept extraction,
   attention weighting, grammar parsing), constructs an enriched system prompt,
   then delegates generation to an external LLM backend (Ollama, vLLM, etc.).
   The LLM output is post-processed back through OPS for formatting.

This makes SynthoLM both a standalone "regex LLM" and an orchestration layer
that augments real LLMs with SYNTHOS's structural intelligence.
"""

from __future__ import annotations

import re
import time
import json
import hashlib
from dataclasses import dataclass, field
from enum import Enum
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple

try:
    import httpx
except ImportError:
    httpx = None  # type: ignore[assignment]

from synthos.layers.lpe import PrimitiveRegistry, PrimitiveType
from synthos.layers.gpl import GeometricParseLattice
from synthos.layers.scm import SemanticConstructManifold
from synthos.layers.tam import TopologicalAttentionMesh
from synthos.layers.rge import RecursiveGrammarEngine
from synthos.layers.scf import StateCrystallizationField, LayerType
from synthos.layers.ops import OutputProjectionSurface, TemplateType
from synthos.utils.config import SynthosConfig, load_config, ModelEndpoint
from synthos.utils.log import get_logger, VerboseLevel
from synthos.tools.executor import ToolExecutor
from synthos.tools.generators import FileGenerator
from synthos.tools.batch import BatchProcessor
from synthos.tools.filesystem import FileSystemTools
from synthos.tools.planner import TaskPlanner, ExecutionPlan, PlanStep
from synthos.lm.distill import (
    build_chain_of_thought, KnowledgeStore,
    generate_candidates, self_consistency_check,
)
from synthos.lm.composer import (
    ResponseComposer, ascii_box, ascii_tree, ascii_table, ascii_progress,
)
from synthos.lm.agent import AgenticLoop, reason_about
from synthos.lm.search import search_and_reason, needs_web_search


class ModelMode(Enum):
    PURE_SYNTAX = "pure_syntax"
    HYBRID = "hybrid"


@dataclass
class GenerationConfig:
    """Controls generation behaviour."""
    mode: ModelMode = ModelMode.PURE_SYNTAX
    max_tokens: int = 512
    max_words: int = 4000              # default response length in words
    max_search_queries: int = 12       # number of search queries for deep research
    max_sources: int = 8               # max sources shown in response
    temperature: float = 0.0          # 0 = deterministic (pure-syntax ignores this)
    model_endpoint: Optional[str] = None   # override config model
    system_prompt: Optional[str] = None    # extra system context
    stream: bool = False
    verbose: bool = False


@dataclass
class GenerationResult:
    """Output of a single generation call."""
    text: str = ""
    mode: str = "pure_syntax"
    tokens_generated: int = 0
    latency_ms: float = 0.0
    pipeline_trace: Dict[str, Any] = field(default_factory=dict)
    model_used: Optional[str] = None

    def to_dict(self) -> Dict[str, Any]:
        return {
            "text": self.text,
            "mode": self.mode,
            "tokens_generated": self.tokens_generated,
            "latency_ms": round(self.latency_ms, 1),
            "model_used": self.model_used,
            "pipeline_trace": {k: str(v)[:200] for k, v in self.pipeline_trace.items()},
        }


# ── Response templates for pure-syntax mode ────────────────────────────────────

_INTENT_PATTERNS: List[Tuple[str, str]] = [
    (r"(?i)(?:what|define|explain|describe|tell me about)\s+(?:is\s+)?(?:a\s+)?(?P<topic>.+)", "explain"),
    (r"(?i)(?:how|why|when|where)\s+(?P<topic>.+)", "explain"),
    (r"(?i)(?:list|enumerate|show|give me)\s+(?P<topic>.+)", "list"),
    (r"(?i)(?:write|generate|create|make)\s+(?:a\s+)?(?:code|function|program|script)\s*(?P<topic>.*)", "code"),
    (r"(?i)^CODE\s+(?P<topic>.+)", "code"),
    (r"(?i)(?:compare|versus|vs|difference between)\s+(?P<topic>.+)", "compare"),
    (r"(?i)(?:summarize|summary|summarise|tldr)\s+(?P<topic>.+)", "summary"),
    (r"(?i)EMIT\s+RESPONSE\s+TYPE:(?P<type>\w+)\s+SUBJECT:(?P<topic>\S+)", "emit"),
    (r"(?i)DEFINE\s+(?P<subject>\w+)\s+AS\s+(?P<value>.+)", "define"),
    (r"(?i)(?P<source>\w+)\s*[⊂→≡⊃↔⊕⊗]\s*(?P<target>\w+)", "relation"),
]

_KNOWLEDGE_FRAGMENTS: Dict[str, str] = {
    "synthos":       "SYNTHOS is a syntax-driven AI architecture where intelligence is encoded in pattern geometry — regex primitives, lattice topology, and substitution chains — rather than neural weights.",
    "attention":     "In SYNTHOS, attention is modelled as pattern intersection area. Eight heads scan input with different regex scopes; overlap between query and key patterns determines relevance.",
    "encryption":    "The Symbolic Topology Cipher (STC) is a 256-bit block cipher with 12 rounds. Each round applies lattice permutation, regex-topology S-box substitution, and Möbius fold.",
    "stc":           "STC supports ECB, CBC (default), and CTR modes. It uses HMAC-SHA256 authentication and PBKDF2 key derivation with 600,000 iterations.",
    "lattice":       "The Geometric Parse Lattice is a 4×4 grid of regex cells. Input is routed spatially from cell to cell based on pattern matches and edge connections.",
    "grammar":       "The Recursive Grammar Engine encodes EBNF productions as recursive regex patterns. It can parse statements, expressions, and geometry definitions.",
    "memory":        "The State Crystallization Field maintains three memory types: short-term (sliding window), long-term (named registers), and episodic (match history stack).",
    "coherence":     "Coherence is measured by testing backreference patterns against the state tensor. A coherence of 1.0 means all backreferences resolve consistently.",
    "neural":        "Unlike neural networks, SYNTHOS uses no floating-point weights, no backpropagation, and no gradient descent. All intelligence is structural.",
    "regex":         "Regular expressions are the atomic cognitive operations of SYNTHOS. Each primitive has an ASCII geometric form: · for dot, ─── for sequence, ○ for groups, ∞ for Kleene star.",
    "transformer":   "Transformers use self-attention with Q·K^T/√d softmax. SYNTHOS replaces this with regex pattern intersection — overlap area as attention weight.",
    "llm":           "Large Language Models predict next tokens via learned distributions. SynthoLM instead routes input through 7 deterministic layers of pattern geometry.",
    "python":        "SYNTHOS is implemented in Python 3.9+. Install with `pip install -e .` to get the `synthos` command system-wide.",
    "cli":           "The CLI supports: synthos (TUI), synthos chat (REPL), synthos encrypt/decrypt, synthos eval, synthos status, synthos config.",
}


def _generate_code_for_topic(topic: str, func_name: str, class_name: str,
                              entries: list, knowledge: str) -> str:
    """Generate real, functional Python code for any topic."""
    topic_lower = topic.lower()
    lines: List[str] = []

    # ── Domain-specific code templates ─────────────────────────────
    if re.search(r"(?i)(machine.?learn|neural|classifier|model|train)", topic_lower):
        lines.extend([
            f'"""',
            f'{topic} — generated by SynthoLM v3.0',
            f'A simple implementation demonstrating core concepts.',
            f'"""',
            f'import random',
            f'import math',
            f'from typing import List, Tuple',
            f'',
            f'',
            f'class {class_name}:',
            f'    """A minimal {topic} implementation."""',
            f'',
            f'    def __init__(self, learning_rate: float = 0.01, epochs: int = 100):',
            f'        self.lr = learning_rate',
            f'        self.epochs = epochs',
            f'        self.weights: List[float] = []',
            f'        self.bias = 0.0',
            f'',
            f'    def fit(self, X: List[List[float]], y: List[float]) -> None:',
            f'        """Train on the given dataset."""',
            f'        n_features = len(X[0]) if X else 0',
            f'        self.weights = [random.uniform(-0.5, 0.5) for _ in range(n_features)]',
            f'        self.bias = 0.0',
            f'',
            f'        for epoch in range(self.epochs):',
            f'            total_loss = 0.0',
            f'            for xi, yi in zip(X, y):',
            f'                pred = self.predict_one(xi)',
            f'                error = yi - pred',
            f'                total_loss += error ** 2',
            f'                for j in range(n_features):',
            f'                    self.weights[j] += self.lr * error * xi[j]',
            f'                self.bias += self.lr * error',
            f'            if (epoch + 1) % 20 == 0:',
            f'                print(f"  Epoch {{epoch+1}}: loss={{total_loss:.4f}}")',
            f'',
            f'    def predict_one(self, x: List[float]) -> float:',
            f'        """Predict a single sample."""',
            f'        return sum(w * xi for w, xi in zip(self.weights, x)) + self.bias',
            f'',
            f'    def predict(self, X: List[List[float]]) -> List[float]:',
            f'        """Predict on a batch."""',
            f'        return [self.predict_one(x) for x in X]',
            f'',
            f'',
            f'if __name__ == "__main__":',
            f'    # Demo: learn XOR-like pattern',
            f'    X = [[0,0], [0,1], [1,0], [1,1]]',
            f'    y = [0, 1, 1, 0]',
            f'    model = {class_name}(learning_rate=0.1, epochs=100)',
            f'    model.fit(X, y)',
            f'    print("Predictions:", model.predict(X))',
        ])

    elif re.search(r"(?i)(sort|search|algorithm|tree|graph|linked.?list|stack|queue)", topic_lower):
        lines.extend([
            f'"""',
            f'{topic} — generated by SynthoLM v3.0',
            f'"""',
            f'from typing import Any, List, Optional',
            f'',
            f'',
            f'def {func_name}(data: List[Any]) -> List[Any]:',
            f'    """Implement {topic} algorithm."""',
            f'    if len(data) <= 1:',
            f'        return data',
            f'    mid = len(data) // 2',
            f'    left = {func_name}(data[:mid])',
            f'    right = {func_name}(data[mid:])',
            f'    return _merge(left, right)',
            f'',
            f'',
            f'def _merge(left: List[Any], right: List[Any]) -> List[Any]:',
            f'    """Merge two sorted lists."""',
            f'    result: List[Any] = []',
            f'    i = j = 0',
            f'    while i < len(left) and j < len(right):',
            f'        if left[i] <= right[j]:',
            f'            result.append(left[i]); i += 1',
            f'        else:',
            f'            result.append(right[j]); j += 1',
            f'    result.extend(left[i:])',
            f'    result.extend(right[j:])',
            f'    return result',
            f'',
            f'',
            f'if __name__ == "__main__":',
            f'    data = [38, 27, 43, 3, 9, 82, 10]',
            f'    print(f"Input:  {{data}}")',
            f'    print(f"Sorted: {{{func_name}(data)}}")',
        ])

    elif re.search(r"(?i)(attention|transformer|embed|token)", topic_lower):
        lines.extend([
            f'"""',
            f'{topic} — generated by SynthoLM v3.0',
            f'',
            f'Demonstrates the attention mechanism concept:',
            f'  score = Q . K^T / sqrt(d_k)',
            f'  weights = softmax(score)',
            f'  output = weights . V',
            f'"""',
            f'import math',
            f'from typing import List',
            f'',
            f'',
            f'def softmax(x: List[float]) -> List[float]:',
            f'    """Compute softmax over a vector."""',
            f'    max_x = max(x)',
            f'    exps = [math.exp(xi - max_x) for xi in x]',
            f'    total = sum(exps)',
            f'    return [e / total for e in exps]',
            f'',
            f'',
            f'def dot(a: List[float], b: List[float]) -> float:',
            f'    """Dot product of two vectors."""',
            f'    return sum(ai * bi for ai, bi in zip(a, b))',
            f'',
            f'',
            f'def {func_name}(query: List[float], keys: List[List[float]],',
            f'    {"" * len(func_name)}values: List[List[float]]) -> List[float]:',
            f'    """Single-head attention over key-value pairs."""',
            f'    d_k = len(query)',
            f'    scores = [dot(query, k) / math.sqrt(d_k) for k in keys]',
            f'    weights = softmax(scores)',
            f'    d_v = len(values[0])',
            f'    output = [0.0] * d_v',
            f'    for w, v in zip(weights, values):',
            f'        for i in range(d_v):',
            f'            output[i] += w * v[i]',
            f'    return output',
            f'',
            f'',
            f'if __name__ == "__main__":',
            f'    Q = [1.0, 0.5, 0.0]',
            f'    K = [[1,0,0], [0,1,0], [0,0,1], [1,1,0]]',
            f'    V = [[1,0], [0,1], [1,1], [0.5,0.5]]',
            f'    out = {func_name}(Q, K, V)',
            f'    print(f"Attention output: {{[round(x,3) for x in out]}}")',
        ])

    elif re.search(r"(?i)(encrypt|cipher|crypto|hash|security)", topic_lower):
        lines.extend([
            f'"""',
            f'{topic} — generated by SynthoLM v3.0',
            f'A simple substitution cipher demonstration.',
            f'"""',
            f'import hashlib',
            f'',
            f'',
            f'def encrypt(plaintext: str, key: str) -> str:',
            f'    """XOR-based encryption with key derivation."""',
            f'    key_hash = hashlib.sha256(key.encode()).digest()',
            f'    encrypted = []',
            f'    for i, ch in enumerate(plaintext):',
            f'        encrypted.append(chr(ord(ch) ^ key_hash[i % len(key_hash)]))',
            f'    return encrypted_hex(encrypted)',
            f'',
            f'',
            f'def encrypted_hex(chars: list) -> str:',
            f'    return "".join(f"{{ord(c):02x}}" for c in chars)',
            f'',
            f'',
            f'def decrypt(hex_text: str, key: str) -> str:',
            f'    """Decrypt XOR-encrypted hex string."""',
            f'    key_hash = hashlib.sha256(key.encode()).digest()',
            f'    data = bytes.fromhex(hex_text)',
            f'    return "".join(chr(b ^ key_hash[i % len(key_hash)]) for i, b in enumerate(data))',
            f'',
            f'',
            f'if __name__ == "__main__":',
            f'    msg = "Hello from SYNTHOS!"',
            f'    key = "secret-key-2024"',
            f'    enc = encrypt(msg, key)',
            f'    print(f"Encrypted: {{enc}}")',
            f'    print(f"Decrypted: {{decrypt(enc, key)}}")',
        ])

    elif re.search(r"(?i)(web|server|api|flask|fastapi|http|rest)", topic_lower):
        lines.extend([
            f'"""',
            f'{topic} — generated by SynthoLM v3.0',
            f'A minimal HTTP server example.',
            f'"""',
            f'from http.server import HTTPServer, BaseHTTPRequestHandler',
            f'import json',
            f'',
            f'',
            f'class {class_name}Handler(BaseHTTPRequestHandler):',
            f'    def do_GET(self):',
            f'        self.send_response(200)',
            f'        self.send_header("Content-Type", "application/json")',
            f'        self.end_headers()',
            f'        response = {{"status": "ok", "service": "{topic}"}}',
            f'        self.wfile.write(json.dumps(response).encode())',
            f'',
            f'    def do_POST(self):',
            f'        length = int(self.headers.get("Content-Length", 0))',
            f'        body = json.loads(self.rfile.read(length)) if length else {{}}',
            f'        self.send_response(200)',
            f'        self.send_header("Content-Type", "application/json")',
            f'        self.end_headers()',
            f'        result = {{"received": body, "processed": True}}',
            f'        self.wfile.write(json.dumps(result).encode())',
            f'',
            f'',
            f'if __name__ == "__main__":',
            f'    server = HTTPServer(("localhost", 8080), {class_name}Handler)',
            f'    print("Server running on http://localhost:8080")',
            f'    server.serve_forever()',
        ])

    else:
        # Generic but functional: produce a real class with methods
        desc = ""
        if entries:
            desc = entries[0].summary
        elif knowledge:
            desc = knowledge.split(".")[0].strip() + "."
        else:
            desc = f"A utility module for {topic}."

        lines.extend([
            f'"""',
            f'{topic} — generated by SynthoLM v3.0',
            f'',
            f'{desc}',
            f'"""',
            f'from dataclasses import dataclass, field',
            f'from typing import Any, Dict, List, Optional',
            f'',
            f'',
            f'@dataclass',
            f'class {class_name}:',
            f'    """Represents a {topic} entity."""',
            f'    name: str = ""',
            f'    data: Dict[str, Any] = field(default_factory=dict)',
            f'    tags: List[str] = field(default_factory=list)',
            f'',
            f'    def process(self) -> str:',
            f'        """Process this {topic} and return a summary."""',
            f'        parts = [f"{{self.name}}: {{len(self.data)}} fields"]',
            f'        if self.tags:',
            f'            parts.append(f"tags={{self.tags}}")',
            f'        return " | ".join(parts)',
            f'',
            f'    def to_dict(self) -> Dict[str, Any]:',
            f'        """Serialize to dictionary."""',
            f'        return {{"name": self.name, "data": self.data, "tags": self.tags}}',
            f'',
            f'    @classmethod',
            f'    def from_dict(cls, d: Dict[str, Any]) -> "{class_name}":',
            f'        return cls(name=d.get("name",""), data=d.get("data",{{}}), tags=d.get("tags",[]))',
            f'',
            f'',
            f'def {func_name}_pipeline(items: List[{class_name}]) -> List[str]:',
            f'    """Run a processing pipeline on a list of {topic} items."""',
            f'    results = []',
            f'    for item in items:',
            f'        result = item.process()',
            f'        results.append(result)',
            f'        print(f"  Processed: {{result}}")',
            f'    return results',
            f'',
            f'',
            f'if __name__ == "__main__":',
            f'    items = [',
            f'        {class_name}(name="alpha", data={{"score": 0.9}}, tags=["primary"]),',
            f'        {class_name}(name="beta", data={{"score": 0.7, "active": True}}),',
            f'    ]',
            f'    results = {func_name}_pipeline(items)',
            f'    print(f"\\nProcessed {{len(results)}} items.")',
        ])

    return "\n".join(lines)


class SynthoLM:
    """
    Syntax-based language model.

    Parameters
    ----------
    config : SynthosConfig | None
        System configuration.
    verbose : VerboseLevel
        Logging verbosity.
    """

    def __init__(self, config: Optional[SynthosConfig] = None, verbose: VerboseLevel = VerboseLevel.NORMAL,
                 workspace: Optional[str] = None):
        self.config = config or load_config()
        self.log = get_logger("synthos.lm", verbose)
        self.verbose = verbose

        # Pipeline layers
        self.lpe = PrimitiveRegistry()
        self.gpl = GeometricParseLattice()
        self.scm = SemanticConstructManifold()
        self.tam = TopologicalAttentionMesh()
        self.rge = RecursiveGrammarEngine()
        self.scf = StateCrystallizationField()
        self.ops = OutputProjectionSurface()

        # Tool system — filesystem, generators, batch processor, planner
        from pathlib import Path
        ws = Path(workspace) if workspace else None
        self.tool_executor = ToolExecutor(workspace=ws, safe_mode=(workspace is None))
        self.file_generator = FileGenerator(self.tool_executor)
        self.fs = FileSystemTools(self.tool_executor)
        self.batch = BatchProcessor(self.tool_executor, self.file_generator)
        self.planner = TaskPlanner(self.tool_executor, self.file_generator)

        # Distillation — knowledge store for retrieval-augmented generation
        self.knowledge = KnowledgeStore()
        self.composer = ResponseComposer(self.knowledge)

        # Conversation context
        self.context: List[Dict[str, str]] = []
        self.turn_count = 0

        self.log.info("SynthoLM engine initialised (workspace: %s)", self.tool_executor.workspace)

    # ── Public API ─────────────────────────────────────────────────────────

    def generate(self, prompt: str, config: Optional[GenerationConfig] = None) -> GenerationResult:
        """
        Generate a response for *prompt*.

        In pure-syntax mode the 7-layer pipeline produces the response
        directly.  In hybrid mode the pipeline enriches the prompt, sends
        it to an LLM backend, and post-processes the result.
        """
        cfg = config or GenerationConfig()
        t0 = time.perf_counter()
        trace = {}

        # 0a. Try instant knowledge answer first (math, conversions, facts)
        instant = reason_about(prompt, max_words=cfg.max_words,
                               max_search_queries=cfg.max_search_queries,
                               max_sources=cfg.max_sources)
        if instant:
            result = GenerationResult(
                text=instant,
                mode="knowledge",
                tokens_generated=len(instant.split()),
                model_used="SynthoLM-knowledge",
            )
            result.latency_ms = (time.perf_counter() - t0) * 1000
            self._update_context(prompt, result)
            return result

        # 0b. Web search with reasoning pre-generation
        if needs_web_search(prompt):
            search_result = search_and_reason(prompt, progress_callback=self._agent_progress)
            result = GenerationResult(
                text=search_result.answer,
                mode="search",
                tokens_generated=len(search_result.answer.split()),
                model_used="SynthoLM-search",
                pipeline_trace={"intent": search_result.plan.intent,
                                "sub_queries": search_result.plan.sub_questions,
                                "sources": len(search_result.sources)},
            )
            result.latency_ms = (time.perf_counter() - t0) * 1000
            self._update_context(prompt, result)
            return result

        # 1. Check if this is a complex goal needing agentic loop (BEFORE planner)
        has_actions = False
        if self._needs_agentic_loop(prompt):
            ws = str(self._workspace) if hasattr(self, '_workspace') else str(Path.home() / "synthos_workspace")
            agent = AgenticLoop(workspace=ws, progress_callback=self._agent_progress)
            agent_result = agent.solve(prompt)
            result = GenerationResult(
                text=agent_result.answer,
                mode="agentic",
                tokens_generated=len(agent_result.answer.split()),
                model_used="SynthoLM-agent",
                pipeline_trace={"iterations": agent_result.total_iterations,
                                "success": agent_result.success},
            )

        else:
            # 2. Try the universal planner — handles simple actionable requests
            exec_plan = self.planner.plan(prompt)
            has_actions = any(s.action is not None for s in exec_plan.steps)

            if has_actions:
                auto = (not exec_plan.needs_approval) or getattr(self, '_auto_approve', False)
                response_text, _ = self.planner.plan_and_execute(prompt, auto_approve=auto)
                result = GenerationResult(
                    text=response_text,
                    mode="tool_action",
                    tokens_generated=len(response_text.split()),
                    model_used="SynthoLM-planner",
                    pipeline_trace={"plan": exec_plan.to_dict()},
                )
            else:
                # 3. Pure-syntax pipeline
                trace = self._analyse(prompt)
                if cfg.mode == ModelMode.HYBRID and self.config.models:
                    result = self._generate_hybrid(prompt, cfg, trace)
                else:
                    result = self._generate_pure(prompt, cfg, trace)
                result.pipeline_trace = trace

        result.latency_ms = (time.perf_counter() - t0) * 1000
        self._update_context(prompt, result)
        return result

    def _update_context(self, prompt: str, result: GenerationResult) -> None:
        """Update conversation context and episodic memory."""
        self.context.append({"role": "user", "content": prompt})
        self.context.append({"role": "assistant", "content": result.text})
        self.turn_count += 1
        self.scf.memory_lattice.add_episode({
            "turn": self.turn_count,
            "prompt": prompt[:100],
            "response_len": len(result.text),
        })

    def chat(self, message: str, config: Optional[GenerationConfig] = None) -> str:
        """Convenience wrapper that returns just the text."""
        return self.generate(message, config).text

    # ── Agentic loop helpers ──────────────────────────────────────────────

    _AGENTIC_KEYWORDS = frozenset({
        "program", "script", "app", "game", "tool", "bot", "calculator",
        "server", "chatbot", "todo", "guessing", "quiz", "timer", "stopwatch",
        "countdown", "clock", "pomodoro", "password", "generator", "manager",
        "file", "explorer", "browser", "parser", "tester", "converter",
        "processor", "analyzer", "scraper", "database", "encryption",
        "decryption", "cipher", "email", "weather", "forecast",
        "benchmark", "simulation", "experiment", "chart", "graph",
    })

    # Keywords that indicate long-form text, NOT an agentic code task
    _LONGFORM_KEYWORDS = frozenset({
        "essay", "article", "report", "paper", "story", "narrative",
        "guide", "tutorial", "howto", "analysis", "briefing", "blog",
        "review", "summary", "document", "piece",
    })

    # Words that indicate a simple file/dir operation, NOT an agentic task
    _PLANNER_KEYWORDS = frozenset({
        "directory", "folder", "dir", "mkdir", "called", "named",
    })

    def _needs_agentic_loop(self, prompt: str) -> bool:
        """Detect if the prompt requires iterative code generation & execution."""
        p = prompt.lower()
        words = set(re.findall(r"\w+", p))

        # Exclude simple file/dir operations — those go to the planner
        if words & self._PLANNER_KEYWORDS:
            return False

        # Exclude long-form text requests — those go to reason_about
        if words & self._LONGFORM_KEYWORDS:
            return False

        # Check for build/make/create + program-type noun
        if re.search(r"(?i)(build|make|create|write|develop|implement|generate)\s+", p):
            if words & self._AGENTIC_KEYWORDS:
                return True

        # Explicit patterns
        return bool(re.search(
            r"(?i)(build\s+(?:me\s+)?(?:a|an)\s+\w+"
            r"|write\s+(?:me\s+)?(?:a|an)\s+\w+"
            r"|make\s+(?:me\s+)?(?:a|an?\s+)?\w+"
            r"|calculate\s+.{10,}|compute\s+.{10,}"
            r"|solve\s+.{10,}"
            r"|run\s+(?:a\s+)?(?:test|benchmark|simulation|experiment)"
            r"|implement\s+(?:a\s+)?\w+|develop\s+(?:a\s+)?\w+"
            r"|code\s+(?:a|an)\s+\w+)",
            prompt,
        ))

    def _agent_progress(self, msg: str) -> None:
        """Progress callback from the agentic loop (used by TUI)."""
        self.log.info(msg)

    # ── Pipeline analysis ──────────────────────────────────────────────────

    def _analyse(self, prompt: str) -> Dict[str, Any]:
        """Run all 7 layers and collect diagnostics."""
        trace: Dict[str, Any] = {}

        # L0 LPE
        prim_hits = 0
        for pid, p in self.lpe.primitives.items():
            try:
                if re.search(p.regex, prompt):
                    prim_hits += 1
            except re.error:
                pass
        trace["lpe"] = {"primitives_matched": prim_hits}

        # L1 GPL
        try:
            gpl_res = self.gpl.traverse_lattice(prompt)
            trace["gpl"] = {"success": gpl_res["success"], "path": gpl_res.get("path_taken", [])}
        except Exception:
            trace["gpl"] = {"success": False}

        # L2 SCM
        concepts = self.scm.extract_concepts_from_text(prompt)
        relations = self.scm.extract_relations_from_text(prompt)
        trace["scm"] = {"concepts": len(concepts), "relations": len(relations),
                         "concept_ids": [c.concept_id for c in concepts[:5]]}

        # L3 TAM
        try:
            mha = self.tam.multi_head_attention(prompt)
            trace["tam"] = {"heads_used": mha.get("heads_used", []), "output_len": len(mha.get("concatenated_output", ""))}
        except Exception:
            trace["tam"] = {"heads_used": []}

        # L4 RGE
        matched_rules = []
        for rule in ("STATEMENT", "EXPRESSION", "GEOMETRY"):
            try:
                if self.rge.parse(prompt, rule):
                    matched_rules.append(rule)
            except Exception:
                pass
        trace["rge"] = {"matched_rules": matched_rules}

        # L5 SCF
        words = prompt.split()
        if words:
            m = re.match(r"\w+", words[0])
            if m:
                self.scf.crystallize_match(r"\w+", m, LayerType.LEXICAL)
        coherence = self.scf.compute_coherence()
        trace["scf"] = {"coherence": coherence}

        # L6 OPS — deferred to generation step
        trace["ops"] = {}

        return trace

    # ── Pure-syntax generation ─────────────────────────────────────────────

    def _generate_pure(self, prompt: str, cfg: GenerationConfig, trace: Dict[str, Any]) -> GenerationResult:
        """Generate using only the SYNTHOS pipeline — no external model."""

        # 1. Detect intent
        intent, topic = self._detect_intent(prompt)

        # 2. Gather knowledge
        knowledge = self._retrieve_knowledge(prompt, topic)

        # 3. Build response based on intent
        if intent == "emit":
            response = self._handle_emit(prompt)
        elif intent == "define":
            response = self._handle_define(prompt)
        elif intent == "relation":
            response = self._handle_relation(prompt, trace)
        elif intent == "list":
            response = self._handle_list(topic, knowledge)
        elif intent == "code":
            response = self._handle_code(topic, knowledge)
        elif intent == "compare":
            response = self._handle_compare(topic, knowledge)
        elif intent == "summary":
            response = self._handle_summary(topic, knowledge)
        else:
            response = self._handle_explain(topic, knowledge, trace, prompt)

        # 4. Append pipeline stats if verbose
        if cfg.verbose:
            stats = self._format_trace(trace)
            response = stats + "\n\n" + response

        tokens = len(response.split())
        return GenerationResult(
            text=response,
            mode="pure_syntax",
            tokens_generated=tokens,
            model_used="SynthoLM-pure",
        )

    def _detect_intent(self, prompt: str) -> Tuple[str, str]:
        for pattern, intent in _INTENT_PATTERNS:
            m = re.search(pattern, prompt)
            if m:
                topic = m.group("topic") if "topic" in m.groupdict() else prompt
                return intent, topic.strip()
        return "explain", prompt.strip()

    def _retrieve_knowledge(self, prompt: str, topic: str) -> str:
        """Find relevant knowledge using the distillation knowledge store."""
        combined = prompt + " " + topic
        entries = self.knowledge.query(combined, top_k=3)
        if entries:
            return self.knowledge.distill(entries)
        # Fallback to static fragments
        combined_lower = combined.lower()
        hits: List[str] = []
        for key, fragment in _KNOWLEDGE_FRAGMENTS.items():
            if key in combined_lower or any(w in combined_lower for w in key.split("_")):
                hits.append(fragment)
        return " ".join(hits[:3]) if hits else ""

    def _handle_emit(self, prompt: str) -> str:
        try:
            processed = self.ops.process_input(prompt, "intent_to_language")
            if "RENDER_" in processed:
                return self.ops.execute_render_command(processed)
            return processed
        except Exception:
            return self.ops.render_template(TemplateType.PARAGRAPH, {
                "TOPIC": "SYNTHOS Response",
                "BODY": f"Processed EMIT command from: {prompt[:60]}"
            })

    def _handle_define(self, prompt: str) -> str:
        m = re.search(r"(?i)DEFINE\s+(?P<subject>\w+)\s+AS\s+(?P<value>.+)", prompt)
        if m:
            subj, val = m.group("subject"), m.group("value")
            self.scf.memory_lattice.store_long_term(subj, val)
            return self.ops.render_template(TemplateType.PARAGRAPH, {
                "TOPIC": f"Definition: {subj}",
                "BODY": f"{subj} is now defined as {val}. Stored in long-term memory register '{subj}'."
            })
        return "Could not parse DEFINE statement."

    def _handle_relation(self, prompt: str, trace: Dict[str, Any]) -> str:
        concepts = trace.get("scm", {}).get("concept_ids", [])
        return self.ops.render_template(TemplateType.PARAGRAPH, {
            "TOPIC": "Semantic Relation",
            "BODY": f"Extracted {len(concepts)} concepts from: {prompt}. "
                    f"Concepts: {', '.join(concepts[:5]) if concepts else 'none detected'}. "
                    f"Relations bound in the Semantic Construct Manifold."
        })

    def _handle_list(self, topic: str, knowledge: str) -> str:
        return self.composer.compose_list(topic, knowledge)

    def _handle_code(self, topic: str, knowledge: str) -> str:
        # Generate real, functional code based on the topic
        func_name = re.sub(r'[^a-z0-9_]', '_', topic.lower()[:30]).strip('_') or 'main'
        class_name = "".join(w.capitalize() for w in func_name.split("_"))
        entries = self.knowledge.query(topic, top_k=2)

        snippet = _generate_code_for_topic(topic, func_name, class_name, entries, knowledge)
        return ascii_box(f"Code: {topic[:40]}", snippet, width=72)

    def _handle_compare(self, topic: str, knowledge: str) -> str:
        return self.composer.compose_compare(topic, knowledge)

    def _handle_summary(self, topic: str, knowledge: str) -> str:
        return self.composer.compose_summary(topic, knowledge)

    def _handle_explain(self, topic: str, knowledge: str, trace: Dict[str, Any],
                         full_prompt: str = "") -> str:
        return self.composer.compose_explain(full_prompt or topic, trace)

    def _format_trace(self, trace: Dict[str, Any]) -> str:
        lines = ["Pipeline Trace:"]
        lines.append(f"  L0 LPE  {trace.get('lpe', {}).get('primitives_matched', 0)} primitives")
        gpl = trace.get("gpl", {})
        lines.append(f"  L1 GPL  {'→'.join(gpl.get('path', [])[:4]) or 'no path'}")
        scm = trace.get("scm", {})
        lines.append(f"  L2 SCM  {scm.get('concepts', 0)} concepts · {scm.get('relations', 0)} relations")
        tam = trace.get("tam", {})
        lines.append(f"  L3 TAM  {len(tam.get('heads_used', []))} heads fired")
        rge = trace.get("rge", {})
        lines.append(f"  L4 RGE  {', '.join(rge.get('matched_rules', [])) or 'no match'}")
        scf = trace.get("scf", {})
        lines.append(f"  L5 SCF  coherence {scf.get('coherence', 0):.2f}")
        lines.append(f"  L6 OPS  output projected")
        return "\n".join(lines)

    # ── Hybrid generation ──────────────────────────────────────────────────

    def _generate_hybrid(self, prompt: str, cfg: GenerationConfig, trace: Dict[str, Any]) -> GenerationResult:
        """Enrich prompt with pipeline analysis, delegate to LLM, post-process."""

        # Pick model endpoint
        ep = self.config.models[0] if self.config.models else None
        if not ep:
            return self._generate_pure(prompt, cfg, trace)

        # Build enriched system prompt
        sys_prompt = self._build_system_prompt(prompt, trace, cfg.system_prompt)

        # Build messages
        messages = [{"role": "system", "content": sys_prompt}]
        for turn in self.context[-10:]:  # last 5 turns
            messages.append(turn)
        messages.append({"role": "user", "content": prompt})

        # Call LLM
        try:
            response_text = self._call_llm(ep, messages, cfg.max_tokens, cfg.temperature)
        except Exception as exc:
            self.log.warning("Hybrid LLM call failed: %s — falling back to pure-syntax", exc)
            return self._generate_pure(prompt, cfg, trace)

        # Post-process through OPS
        try:
            processed = self.ops.process_input(response_text, "default_chain") if hasattr(self.ops, 'substitution_chains') else response_text
        except Exception:
            processed = response_text

        tokens = len(response_text.split())
        return GenerationResult(
            text=processed,
            mode="hybrid",
            tokens_generated=tokens,
            model_used=ep.name,
        )

    def _build_system_prompt(self, prompt: str, trace: Dict[str, Any], extra: Optional[str] = None) -> str:
        """Construct an enriched system prompt from pipeline analysis."""
        concepts = trace.get("scm", {}).get("concept_ids", [])
        rules = trace.get("rge", {}).get("matched_rules", [])
        coherence = trace.get("scf", {}).get("coherence", 0)

        parts = [
            "You are SYNTHOS, a syntax-driven AI assistant. Respond clearly and helpfully.",
            f"Pipeline analysis: {trace.get('lpe', {}).get('primitives_matched', 0)} regex primitives matched.",
        ]
        if concepts:
            parts.append(f"Detected concepts: {', '.join(concepts[:5])}.")
        if rules:
            parts.append(f"Grammar rules matched: {', '.join(rules)}.")
        parts.append(f"State coherence: {coherence:.2f}.")

        # Add relevant knowledge
        _, topic = self._detect_intent(prompt)
        knowledge = self._retrieve_knowledge(prompt, topic)
        if knowledge:
            parts.append(f"Relevant context: {knowledge}")

        if extra:
            parts.append(extra)

        return " ".join(parts)

    def _call_llm(self, ep: ModelEndpoint, messages: List[Dict[str, str]], max_tokens: int, temperature: float) -> str:
        """Synchronous LLM call (Ollama or OpenAI-compat)."""
        if ep.provider == "ollama":
            return self._call_ollama(ep, messages, max_tokens, temperature)
        else:
            return self._call_openai_compat(ep, messages, max_tokens, temperature)

    def _call_ollama(self, ep: ModelEndpoint, messages: List[Dict[str, str]], max_tokens: int, temp: float) -> str:
        url = f"{ep.base_url}/api/chat"
        body = {
            "model": ep.model_id or ep.name,
            "messages": messages,
            "stream": False,
            "options": {"num_predict": max_tokens, "temperature": temp},
        }
        r = httpx.post(url, json=body, timeout=60.0)
        r.raise_for_status()
        return r.json().get("message", {}).get("content", "")

    def _call_openai_compat(self, ep: ModelEndpoint, messages: List[Dict[str, str]], max_tokens: int, temp: float) -> str:
        url = f"{ep.base_url}/v1/chat/completions"
        body = {
            "model": ep.model_id or ep.name,
            "messages": messages,
            "max_tokens": max_tokens,
            "temperature": temp,
        }
        r = httpx.post(url, json=body, timeout=60.0)
        r.raise_for_status()
        return r.json()["choices"][0]["message"]["content"]
