"""
Response Composer — intelligent natural language + ASCII output generation.

Weaves together chain-of-thought reasoning, distilled knowledge, and
structured ASCII formatting to produce intelligent, readable responses.

Output modes:
  - Explain: multi-paragraph reasoning with knowledge integration
  - Compare: side-by-side analysis with ASCII table
  - List: organized items with context
  - Summary: condensed knowledge with key takeaways
  - Code: annotated code blocks with explanation
  - Action: tool execution results with ASCII file trees
"""

from __future__ import annotations

import re
import textwrap
from typing import Any, Dict, List, Optional, Tuple

from synthos.lm.distill import (
    build_chain_of_thought, KnowledgeStore, ChainOfThought,
    generate_candidates, self_consistency_check,
)


# ═══════════════════════════════════════════════════════════════════════════════
# ASCII formatting primitives
# ═══════════════════════════════════════════════════════════════════════════════

def ascii_box(title: str, body: str, width: int = 60) -> str:
    """Render text inside an ASCII box."""
    inner_w = width - 4
    lines = []
    lines.append("╔" + "═" * (width - 2) + "╗")
    # Title
    pad_title = title.center(inner_w)
    lines.append("║ " + pad_title + " ║")
    lines.append("╠" + "═" * (width - 2) + "╣")
    # Body
    for raw_line in body.split("\n"):
        wrapped = textwrap.wrap(raw_line, inner_w) if raw_line.strip() else [""]
        for wl in wrapped:
            lines.append("║ " + wl.ljust(inner_w) + " ║")
    lines.append("╚" + "═" * (width - 2) + "╝")
    return "\n".join(lines)


def ascii_tree(name: str, structure: Dict[str, Any], prefix: str = "") -> str:
    """Render a file/directory tree in ASCII."""
    lines = []
    if not prefix:
        lines.append(f"📁 {name}/")
    items = list(structure.items())
    for i, (key, val) in enumerate(items):
        is_last = (i == len(items) - 1)
        connector = "└── " if is_last else "├── "
        child_prefix = prefix + ("    " if is_last else "│   ")

        if isinstance(val, dict):
            lines.append(f"{prefix}{connector}📁 {key}/")
            lines.append(ascii_tree("", val, child_prefix))
        else:
            # File — show icon based on extension
            icon = _file_icon(key)
            lines.append(f"{prefix}{connector}{icon} {key}")
    return "\n".join(lines)


def ascii_table(headers: List[str], rows: List[List[str]], col_widths: Optional[List[int]] = None) -> str:
    """Render a simple ASCII table."""
    if not col_widths:
        col_widths = []
        for i, h in enumerate(headers):
            max_w = len(h)
            for row in rows:
                if i < len(row):
                    max_w = max(max_w, len(str(row[i])))
            col_widths.append(min(max_w + 2, 40))

    def _row(cells):
        parts = []
        for i, cell in enumerate(cells):
            w = col_widths[i] if i < len(col_widths) else 20
            parts.append(f" {str(cell):<{w - 1}}")
        return "│" + "│".join(parts) + "│"

    def _sep(char="─", join="┼", left="├", right="┤"):
        parts = [char * w for w in col_widths]
        return left + join.join(parts) + right

    lines = []
    lines.append("┌" + "┬".join("─" * w for w in col_widths) + "┐")
    lines.append(_row(headers))
    lines.append(_sep())
    for row in rows:
        lines.append(_row(row))
    lines.append("└" + "┴".join("─" * w for w in col_widths) + "┘")
    return "\n".join(lines)


def ascii_progress(steps: List[Tuple[str, str]], width: int = 50) -> str:
    """Render a step-by-step progress indicator."""
    lines = []
    for i, (label, status) in enumerate(steps):
        if status == "done":
            icon = "✓"
        elif status == "failed":
            icon = "✗"
        elif status == "running":
            icon = "►"
        else:
            icon = "○"

        connector = "│" if i < len(steps) - 1 else " "
        lines.append(f"  {icon} {label}")
        if i < len(steps) - 1:
            lines.append(f"  {connector}")
    return "\n".join(lines)


def _file_icon(filename: str) -> str:
    ext = filename.rsplit(".", 1)[-1] if "." in filename else ""
    icons = {
        "py": "🐍", "sh": "⚙️", "md": "📝", "txt": "📄",
        "json": "📋", "yaml": "📋", "yml": "📋", "toml": "📋",
        "html": "🌐", "css": "🎨", "js": "📜", "ts": "📜",
    }
    return icons.get(ext, "📄")


# ═══════════════════════════════════════════════════════════════════════════════
# Stop words for subject extraction
# ═══════════════════════════════════════════════════════════════════════════════

_STOP_WORDS = frozenset({
    "the", "a", "an", "is", "are", "was", "were", "be", "been", "being",
    "have", "has", "had", "do", "does", "did", "will", "would", "shall",
    "should", "may", "might", "must", "can", "could", "about", "above",
    "after", "again", "all", "also", "and", "any", "because", "but",
    "by", "for", "from", "how", "if", "in", "into", "it", "its",
    "just", "like", "more", "most", "not", "of", "on", "or", "other",
    "out", "over", "own", "same", "so", "some", "such", "than", "that",
    "then", "there", "these", "this", "those", "through", "to", "too",
    "under", "until", "up", "very", "what", "when", "where", "which",
    "while", "who", "whom", "why", "with", "you", "your", "me", "my",
    "tell", "explain", "describe", "show", "give", "please",
})


def _generate_topic_response(subject: str, full_topic: str, cot: "ChainOfThought") -> str:
    """
    Generate an intelligent natural language response for a topic
    that has no matching knowledge store entry.

    Instead of saying 'I don't know', produces actual content by:
    - Categorizing the topic domain
    - Generating concrete observations
    - Relating it to SYNTHOS concepts where possible
    - Offering actionable next steps
    """
    lines: List[str] = []
    sub_lower = subject.lower()
    sub_title = subject.title()

    # ── Detect domain and generate domain-specific content ─────────
    domain_responses = {
        # Science
        r"(?i)(quantum|physics|relativity|particle|electron|atom|molecule|chemistry|biology|cell|DNA|evolution|science)":
            lambda s: (
                f"{s.title()} is a scientific domain that deals with understanding "
                f"natural phenomena through observation, hypothesis, and experimentation. "
                f"In the context of SYNTHOS, scientific concepts can be modeled as "
                f"pattern-matching problems — where hypotheses are regex patterns and "
                f"experimental data is the input string being matched."
            ),
        # Math
        r"(?i)(math|algebra|calculus|geometry|theorem|equation|number|statistics|probability)":
            lambda s: (
                f"{s.title()} is a foundational discipline built on axioms, definitions, "
                f"and logical proofs. SYNTHOS's grammar engine (RGE) uses similar formal "
                f"systems — EBNF productions that decompose complex structures into "
                f"verifiable components, much like mathematical proofs decompose into lemmas."
            ),
        # Programming / CS
        r"(?i)(programming|software|algorithm|computer|data\s*structure|python|javascript|code|compiler|database|API|web|app)":
            lambda s: (
                f"{s.title()} is closely related to what SYNTHOS does natively. "
                f"The 7-layer pipeline is itself a software architecture, and regex — "
                f"SYNTHOS's core primitive — is fundamental to programming. I can "
                f"generate code, scaffold projects, or create files related to {subject} "
                f"for you. Just ask!"
            ),
        # AI / ML
        r"(?i)(machine\s*learning|neural|deep\s*learning|GPT|LLM|artificial\s*intelligence|AI|model|training|inference)":
            lambda s: (
                f"{s.title()} is a field SYNTHOS contrasts with directly. While most "
                f"AI systems use neural networks with learned weights, SYNTHOS encodes "
                f"intelligence in regex pattern geometry — no training, no gradients, "
                f"no backpropagation. The 7-layer pipeline (LPE→GPL→SCM→TAM→RGE→SCF→OPS) "
                f"achieves attention, memory, and reasoning through structural pattern "
                f"matching alone."
            ),
        # Philosophy / abstract
        r"(?i)(philosophy|consciousness|meaning|truth|ethics|morality|existence|mind|thought|logic)":
            lambda s: (
                f"{s.title()} is a domain of inquiry that asks fundamental questions. "
                f"From SYNTHOS's perspective, reasoning is pattern matching — a thought "
                f"is a regex that matches against the space of possible conclusions. "
                f"The coherence score in the State Crystallization Field (SCF) measures "
                f"how well patterns align, which could be seen as a formal analog of "
                f"logical consistency."
            ),
        # Language / text
        r"(?i)(language|grammar|syntax|semantics|word|sentence|writing|text|NLP|natural\s*language)":
            lambda s: (
                f"{s.title()} is at the heart of what SYNTHOS processes. Every input "
                f"passes through lexical analysis (LPE), structural parsing (GPL+RGE), "
                f"and semantic extraction (SCM). SYNTHOS treats language as a pattern "
                f"matching problem — words are atoms, sentences are sequences, and "
                f"meaning emerges from the geometry of regex overlaps."
            ),
        # Music / art / creative
        r"(?i)(music|art|creative|painting|song|poem|poetry|story|fiction|novel|movie|film|game)":
            lambda s: (
                f"{s.title()} is a creative domain. While SYNTHOS is primarily an "
                f"analytical system, its Output Projection Surface (OPS) can generate "
                f"structured text — including formatted prose, ASCII art, and organized "
                f"content. I can help you create project files, templates, or scripts "
                f"related to {subject}."
            ),
    }

    matched = False
    for pattern, generator in domain_responses.items():
        if re.search(pattern, full_topic):
            lines.append(generator(subject))
            matched = True
            break

    if not matched:
        # Active reasoning about the topic — never deflect
        lines.append(
            f"Let me reason about {sub_title} through pattern analysis."
        )

        # Decompose the topic into meaningful observations
        words = [w for w in re.findall(r"\w+", full_topic.lower())
                 if len(w) > 2 and w not in {"the", "and", "for", "are", "was",
                 "how", "what", "why", "who", "when", "where", "you", "your"}]
        if words:
            lines.append(
                f"The key concepts here are: {', '.join(w.title() for w in words[:5])}. "
                f"Each of these maps to patterns in SYNTHOS's lexical layer (LPE) and "
                f"connects through the semantic construct manifold (SCM) to form a "
                f"relational graph of meaning."
            )

        # Generate a substantive paragraph about the topic
        lines.append(
            f"{sub_title} can be understood through multiple lenses — its definition, "
            f"its relationships to other concepts, its practical applications, and "
            f"its underlying structure. What makes pattern-based reasoning powerful "
            f"is that any concept, no matter how abstract, can be decomposed into "
            f"component patterns and their intersections."
        )

        lines.append(
            f"For deeper analysis, try asking me a specific question about {subject} — "
            f"I can run deep research across multiple web sources, generate code "
            f"related to it, or compare it with other concepts."
        )

    return "\n\n".join(lines)


# ═══════════════════════════════════════════════════════════════════════════════
# Intelligent Response Composer
# ═══════════════════════════════════════════════════════════════════════════════

class ResponseComposer:
    """
    Generates intelligent natural language responses by combining:
    - Chain-of-thought reasoning
    - Knowledge store retrieval
    - Self-consistency checking
    - ASCII formatting for structure
    """

    def __init__(self, knowledge: Optional[KnowledgeStore] = None):
        self.knowledge = knowledge or KnowledgeStore()

    def compose_explain(self, topic: str, trace: Dict[str, Any]) -> str:
        """Generate an intelligent explanation with reasoning and knowledge.

        ALWAYS produces a direct, literal natural language response that
        addresses the user's actual input — never just reasoning skeletons.
        """
        # 1. Retrieve relevant knowledge
        entries = self.knowledge.query(topic, top_k=3)

        # 2. Build chain-of-thought
        cot = build_chain_of_thought(topic)

        # 3. Detect conversational / social input
        t = topic.strip()
        t_lower = t.lower()

        is_greeting = bool(re.match(
            r"(?i)^(hi|hello|hey|greetings|good\s+(morning|afternoon|evening)|yo|sup)\b", t))
        is_social = bool(re.match(
            r"(?i)^(how\s+are\s+you|how('s| is)\s+(it\s+going|everything|life|things|your\s+day)"
            r"|what'?s\s+up|how\s+do\s+you\s+(do|feel)|how\s+you\s+doing"
            r"|are\s+you\s+(ok|okay|good|well|alive|real|sentient|conscious))", t))
        is_thanks = bool(re.match(
            r"(?i)^(thanks?|thank\s+you|thx|ty|cheers|appreciate|nice|cool|great|awesome|good\s+job)", t))
        is_farewell = bool(re.match(
            r"(?i)^(bye|goodbye|see\s+you|later|quit|exit|peace|take\s+care|good\s*night)", t))
        is_question_about_self = bool(re.search(
            r"(?i)(what can you|who are you|what are you|tell me about yourself|what do you do)", t))

        paragraphs: List[str] = []

        # ── Greetings ──────────────────────────────────────────────
        if is_greeting:
            paragraphs.append(
                "Hello! I'm SYNTHOS, a syntax-driven AI assistant. "
                "I process your messages through a 7-layer regex pipeline — "
                "no neural weights, just pattern geometry."
            )
            paragraphs.append(
                "Ask me anything — explain concepts, answer questions, "
                "generate code, create projects, compare topics, encrypt text, "
                "or run commands. What would you like to explore?"
            )
            return "\n\n".join(paragraphs)

        # ── Social / "how are you" ────────────────────────────────
        if is_social:
            paragraphs.append(
                "I'm doing well, thank you for asking! All 7 layers are online "
                "and pattern-matching at full capacity."
            )
            paragraphs.append(
                "I'm SYNTHOS — a regex-driven AI that processes language through "
                "lexical primitives, geometric lattice traversal, semantic graphs, "
                "multi-head attention, grammar parsing, state crystallization, and "
                "output projection. No neural weights — just structured pattern geometry."
            )
            paragraphs.append(
                "What can I help you with? I can answer questions, explain concepts, "
                "generate code, create files, run commands, or have a conversation."
            )
            return "\n\n".join(paragraphs)

        # ── Thanks ────────────────────────────────────────────────
        if is_thanks:
            paragraphs.append(
                "You're welcome! Happy to help. Let me know if there's "
                "anything else you'd like to explore."
            )
            return "\n\n".join(paragraphs)

        # ── Farewell ──────────────────────────────────────────────
        if is_farewell:
            paragraphs.append(
                "Goodbye! It was great chatting with you. "
                "Come back anytime — all 7 layers will be here waiting."
            )
            return "\n\n".join(paragraphs)

        # ── Self-description ───────────────────────────────────────
        if is_question_about_self:
            paragraphs.append(
                "I'm SYNTHOS — a syntax-driven AI architecture where intelligence "
                "is encoded in regex pattern geometry across 7 layers, not neural weights."
            )
            paragraphs.append(
                "My capabilities include:\n"
                "  • Deep research — multi-query web search with synthesized answers\n"
                "  • Long-form writing — essays, reports, articles, guides\n"
                "  • Code generation — full programs with 15+ templates\n"
                "  • File & project scaffolding\n"
                "  • Command execution — shell commands with safety checks\n"
                "  • Encryption — Symbolic Topology Cipher (STC)\n"
                "  • Conversational reasoning about any topic"
            )
            return "\n\n".join(paragraphs)

        # ── Knowledge-grounded response ────────────────────────────
        if entries:
            primary = entries[0]
            # Lead with a direct answer sentence
            paragraphs.append(primary.summary)
            # Add supporting details
            if primary.details:
                for detail in primary.details[:3]:
                    paragraphs.append(f"  • {detail}")

            # Additional related entries
            if len(entries) > 1:
                for e in entries[1:3]:
                    paragraphs.append(e.summary)
                    if e.details:
                        paragraphs.append(f"  • {e.details[0]}")

            # Reasoning trace (compact)
            if cot.steps:
                reasoning = "  →  ".join(s.thought for s in cot.steps)
                paragraphs.append(f"[Reasoning: {reasoning}]")

        else:
            # ── No knowledge match — generate intelligent content anyway ──
            subject_words = [w for w in topic.split() if len(w) > 2
                            and w.lower() not in _STOP_WORDS]
            subject = " ".join(subject_words) if subject_words else topic

            # Build a real response about the subject
            paragraphs.append(_generate_topic_response(subject, topic, cot))

        # Pipeline stats — only shown in verbose mode (trace has verbose flag)
        if trace.get("verbose"):
            coh = trace.get("scf", {}).get("coherence", 0)
            prims = trace.get("lpe", {}).get("primitives_matched", 0)
            concepts = trace.get("scm", {}).get("concepts", 0)
            if prims or concepts:
                paragraphs.append(
                    f"[Analysis: {prims} primitives matched, "
                    f"{concepts} concepts extracted, coherence={coh:.2f}]"
                )

        return "\n\n".join(p for p in paragraphs if p.strip())

    def compose_compare(self, topic: str, knowledge: str) -> str:
        """Generate a comparison with ASCII table."""
        parts = re.split(r"\s+(?:vs|versus|and|or|compared\s+to)\s+", topic, flags=re.I)
        if len(parts) < 2:
            parts = [topic, "(other)"]

        a, b = parts[0].strip(), parts[1].strip()

        # Get knowledge for each
        entries_a = self.knowledge.query(a, top_k=2)
        entries_b = self.knowledge.query(b, top_k=2)

        rows = []
        summary_a = entries_a[0].summary if entries_a else f"{a}: no specific knowledge available"
        summary_b = entries_b[0].summary if entries_b else f"{b}: no specific knowledge available"

        # Build comparison table
        rows.append([a[:18], b[:18]])
        if entries_a and entries_a[0].details:
            for detail in entries_a[0].details[:3]:
                b_detail = entries_b[0].details[len(rows) - 1] if entries_b and entries_b[0].details and len(rows) - 1 < len(entries_b[0].details) else "—"
                rows.append([detail[:38], b_detail[:38]])

        table = ascii_table([a[:18], b[:18]], rows, [20, 20])

        response_parts = [
            f"Comparing {a} and {b}:\n",
            summary_a,
            summary_b,
            "",
            table,
        ]
        if knowledge:
            response_parts.append(f"\n{knowledge}")

        return "\n\n".join(p for p in response_parts if p)

    def compose_list(self, topic: str, knowledge: str) -> str:
        """Generate an organized list with context."""
        items = [s.strip() for s in re.split(r"[,;]|\band\b", topic) if s.strip()]
        if not items:
            items = [topic]

        lines = [f"Here are the items related to '{topic}':\n"]
        for i, item in enumerate(items, 1):
            # Try to get knowledge for each item
            item_entries = self.knowledge.query(item, top_k=1)
            if item_entries:
                lines.append(f"  {i}. {item}")
                lines.append(f"     {item_entries[0].summary[:80]}")
            else:
                lines.append(f"  {i}. {item}")

        if knowledge:
            lines.append(f"\nAdditional context: {knowledge}")

        return "\n".join(lines)

    def compose_summary(self, topic: str, knowledge: str) -> str:
        """Generate a condensed summary with key takeaways."""
        entries = self.knowledge.query(topic, top_k=3)

        parts = []
        if entries:
            parts.append(entries[0].summary)
            if entries[0].details:
                parts.append("\nKey points:")
                for d in entries[0].details:
                    parts.append(f"  • {d}")
        elif knowledge:
            parts.append(knowledge)
        else:
            parts.append(f"Summary of '{topic}': No detailed knowledge available in the current store.")

        if len(entries) > 1:
            parts.append("\nRelated topics:")
            for e in entries[1:]:
                parts.append(f"  • {e.topic}: {e.summary[:60]}...")

        return "\n".join(parts)

    def compose_action_result(self, plan_data: Dict[str, Any],
                               created_files: List[str] = None,
                               created_dirs: List[str] = None) -> str:
        """Generate a rich action result with ASCII tree and progress."""
        steps = plan_data.get("steps", [])
        request = plan_data.get("request", "")

        parts = []

        # Header
        parts.append(f"I understood your request: \"{request}\"")
        parts.append("")

        # Progress indicator
        progress_items = []
        for s in steps:
            status = s.get("status", "pending")
            desc = s.get("thought", s.get("description", ""))
            if len(desc) > 50:
                desc = desc[:50] + "..."
            progress_items.append((desc, "done" if status == "success" else status))

        if progress_items:
            parts.append("Steps executed:")
            parts.append(ascii_progress(progress_items))
            parts.append("")

        # File tree if things were created
        if created_dirs or created_files:
            tree_struct = {}
            all_paths = (created_dirs or []) + (created_files or [])
            for p in all_paths:
                _insert_path(tree_struct, p)

            if tree_struct:
                root_name = list(tree_struct.keys())[0] if len(tree_struct) == 1 else "workspace"
                if len(tree_struct) == 1:
                    parts.append(ascii_tree(root_name, tree_struct[root_name]))
                else:
                    parts.append(ascii_tree("workspace", tree_struct))
                parts.append("")

        # Summary
        done_count = sum(1 for s in steps if s.get("status") == "success")
        fail_count = sum(1 for s in steps if s.get("status") == "error")
        total = len(steps)

        if fail_count == 0 and total > 0:
            parts.append(f"All {total} step(s) completed successfully.")
        elif fail_count > 0:
            parts.append(f"Completed: {done_count}/{total} steps ({fail_count} failed).")
        else:
            parts.append("Task completed.")

        return "\n".join(parts)


def _insert_path(tree: dict, path: str):
    """Insert a file path into a nested dict tree structure."""
    parts = path.strip("/").split("/")
    current = tree
    for i, part in enumerate(parts):
        if i == len(parts) - 1:
            # Leaf — check if it's a file (has extension) or dir
            if "." in part:
                current[part] = ""
            else:
                current.setdefault(part, {})
        else:
            current = current.setdefault(part, {})
