# ═══════════════════════════════════════════════════════════════════════════════════
# SYNTHOS SEMANTIC CONSTRUCT MANIFOLD (SCM) v1.0
# Named captures as semantic nodes with relation edges and context windows
# ═══════════════════════════════════════════════════════════════════════════════════

import re
from typing import Dict, List, Tuple, Optional, Set, Any
from dataclasses import dataclass, field
from enum import Enum
import networkx as nx
from collections import defaultdict

class RelationType(Enum):
    FLOWS_TO = "→"
    SUBSET_OF = "⊂"
    EQUIVALENT_TO = "≡"
    CONTAINS = "⊃"
    BIDIRECTIONAL = "↔"
    COMBINES_WITH = "⊕"
    MODULATES = "⊗"

@dataclass
class ConceptNode:
    """Semantic node representing a concept in the manifold"""
    concept_id: str
    genus: str
    differentia: str
    pattern: str
    captures: Dict[str, str] = field(default_factory=dict)
    abstraction_level: int = 0
    activation: float = 0.0
    
    @property
    def full_concept(self) -> str:
        """Get the full concept string"""
        if self.differentia:
            separator = " " if self.genus[-1].isalnum() else ""
            return f"{self.genus}{separator}{self.differentia}"
        return self.genus
    
    def __str__(self):
        return f"⟦{self.concept_id}: {self.full_concept}⟧"

@dataclass
class RelationEdge:
    """Relation edge between concept nodes"""
    source: str
    target: str
    relation_type: RelationType
    weight: float = 1.0
    bidirectional: bool = False
    
    def __str__(self):
        return f"{self.source} {self.relation_type.value} {self.target}"

@dataclass
class ContextWindow:
    """Context window for attention focus"""
    focus: str
    back_context: str = ""
    fore_context: str = ""
    window_size: int = 512
    
    def __str__(self):
        return f"[←{len(self.back_context)}←]──║ {self.focus} ║──[→{len(self.fore_context)}→]"

class SemanticConstructManifold:
    """Manifold for managing semantic constructs and their relationships"""
    
    def __init__(self):
        self.concepts: Dict[str, ConceptNode] = {}
        self.relations: List[RelationEdge] = []
        self.context_windows: Dict[str, ContextWindow] = {}
        self.graph = nx.DiGraph()
        self.abstraction_hierarchy: Dict[int, List[str]] = defaultdict(list)
        
        # Regex patterns for concept extraction
        self.concept_pattern = re.compile(
            r'(?P<CONCEPT_ID>'
            r'(?P<GENUS>\w+)'                    # What category
            r'[\s\-\_]?'                         # Optional separator
            r'(?P<DIFFERENTIA>\w*)'              # Distinguishing property
            r')'
        )
        
        self.relation_pattern = re.compile(
            r'(?P<SOURCE>\w+)'
            r'(?P<REL_TYPE>'
            r'(?:→|⊂|≡|⊃|↔|⊕|⊗|causes?|implies?|is|has|was|will)'
            r')'
            r'(?P<TARGET>\w+)'
        )
        
        # Context window pattern — Python re doesn't allow variable-width
        # lookbehinds, so we use a capturing-group approach instead:
        #   group 1 = back context (up to 512 chars)
        #   group 2 = focus token
        #   group 3 = fore context (up to 512 chars, via lookahead)
        self.context_pattern = re.compile(
            r'(.{0,512}?)'                        # BACK_CTX (greedy-short)
            r'(?P<FOCUS>\S+)'                     # Current attention focus
            r'(?='
            r'(?P<FORE_CTX>.{0,512})'            # Forward context buffer
            r')'
        )
    
    def add_concept(self, concept: ConceptNode):
        """Add a concept node to the manifold"""
        self.concepts[concept.concept_id] = concept
        self.graph.add_node(concept.concept_id, **concept.__dict__)
        self.abstraction_hierarchy[concept.abstraction_level].append(concept.concept_id)
    
    def add_relation(self, relation: RelationEdge):
        """Add a relation edge between concepts"""
        self.relations.append(relation)
        
        # Add to graph
        self.graph.add_edge(
            relation.source, 
            relation.target,
            relation_type=relation.relation_type.value,
            weight=relation.weight
        )
        
        # Add reverse edge if bidirectional
        if relation.bidirectional or relation.relation_type == RelationType.BIDIRECTIONAL:
            self.graph.add_edge(
                relation.target,
                relation.source,
                relation_type=relation.relation_type.value,
                weight=relation.weight
            )
    
    def extract_concepts_from_text(self, text: str) -> List[ConceptNode]:
        """Extract concepts from text using regex patterns"""
        concepts = []
        
        for match in self.concept_pattern.finditer(text):
            concept_id = match.group('CONCEPT_ID')
            genus = match.group('GENUS')
            differentia = match.group('DIFFERENTIA') or ""
            
            # Create concept node
            concept = ConceptNode(
                concept_id=concept_id,
                genus=genus,
                differentia=differentia,
                pattern=match.group(0),
                captures=match.groupdict(),
                abstraction_level=self._calculate_abstraction_level(genus, differentia)
            )
            
            concepts.append(concept)
            self.add_concept(concept)
        
        return concepts
    
    def extract_relations_from_text(self, text: str) -> List[RelationEdge]:
        """Extract relations from text using regex patterns"""
        relations = []
        
        for match in self.relation_pattern.finditer(text):
            source = match.group('SOURCE')
            target = match.group('TARGET')
            rel_glyph = match.group('REL_TYPE')
            
            # Map glyph to relation type
            relation_type_map = {
                "→": RelationType.FLOWS_TO,
                "⊂": RelationType.SUBSET_OF,
                "≡": RelationType.EQUIVALENT_TO,
                "⊃": RelationType.CONTAINS,
                "↔": RelationType.BIDIRECTIONAL,
                "⊕": RelationType.COMBINES_WITH,
                "⊗": RelationType.MODULATES,
                "causes": RelationType.FLOWS_TO,
                "implies": RelationType.FLOWS_TO,
                "is": RelationType.EQUIVALENT_TO,
                "has": RelationType.CONTAINS,
                "was": RelationType.EQUIVALENT_TO,
                "will": RelationType.FLOWS_TO
            }
            
            relation_type = relation_type_map.get(rel_glyph, RelationType.FLOWS_TO)
            
            relation = RelationEdge(
                source=source,
                target=target,
                relation_type=relation_type,
                bidirectional=(relation_type == RelationType.BIDIRECTIONAL)
            )
            
            relations.append(relation)
            self.add_relation(relation)
        
        return relations
    
    def extract_context_windows(self, text: str, focus_size: int = 50) -> Dict[str, ContextWindow]:
        """Extract context windows from text"""
        windows = {}
        
        for match in self.context_pattern.finditer(text):
            focus = match.group('FOCUS')
            back_ctx = match.group('BACK_CTX') or ""
            fore_ctx = match.group('FORE_CTX') or ""
            
            # Create context window
            window = ContextWindow(
                focus=focus,
                back_context=back_ctx,
                fore_context=fore_ctx,
                window_size=512
            )
            
            windows[focus] = window
            self.context_windows[focus] = window
        
        return windows
    
    def _calculate_abstraction_level(self, genus: str, differentia: str) -> int:
        """Calculate abstraction level based on concept properties"""
        # Higher abstraction = fewer specific properties
        base_level = 0
        
        # Genus abstraction (more general = higher level)
        genus_abstraction = {
            'entity': 5, 'object': 4, 'thing': 3, 'item': 2, 'instance': 1,
            'concept': 6, 'idea': 5, 'notion': 4, 'thought': 3,
            'process': 4, 'action': 3, 'activity': 2, 'operation': 1,
            'property': 3, 'attribute': 2, 'quality': 1
        }
        
        base_level += genus_abstraction.get(genus.lower(), 2)
        
        # Differentia reduces abstraction (more specific = lower level)
        if differentia:
            base_level -= len(differentia.split('_')) * 0.5
        
        return max(0, int(base_level))
    
    def find_related_concepts(self, concept_id: str, relation_types: Optional[List[RelationType]] = None) -> List[str]:
        """Find concepts related to a given concept"""
        if concept_id not in self.concepts:
            return []
        
        related = []
        
        for relation in self.relations:
            if relation.source == concept_id:
                if relation_types is None or relation.relation_type in relation_types:
                    related.append(relation.target)
            elif relation.target == concept_id:
                if relation_types is None or relation.relation_type in relation_types:
                    related.append(relation.source)
        
        return related
    
    def get_abstraction_path(self, concept_id: str) -> List[str]:
        """Get abstraction path from most abstract to most specific"""
        if concept_id not in self.concepts:
            return []
        
        concept = self.concepts[concept_id]
        path = []
        
        # Build path from highest to lowest abstraction
        for level in sorted(self.abstraction_hierarchy.keys(), reverse=True):
            for cid in self.abstraction_hierarchy[level]:
                if cid == concept_id:
                    path.append(cid)
                elif self._is_ancestor_of(cid, concept_id):
                    path.append(cid)
        
        return path
    
    def _is_ancestor_of(self, ancestor_id: str, descendant_id: str) -> bool:
        """Check if one concept is an ancestor of another"""
        try:
            return nx.has_path(self.graph, ancestor_id, descendant_id)
        except:
            return False
    
    def activate_concept(self, concept_id: str, activation: float = 1.0):
        """Activate a concept and propagate activation through relations"""
        if concept_id not in self.concepts:
            return
        
        self.concepts[concept_id].activation = activation
        
        # Propagate to related concepts
        for relation in self.relations:
            if relation.source == concept_id:
                if relation.target in self.concepts:
                    # Reduce activation based on relation weight
                    propagated_activation = activation * relation.weight * 0.8
                    self.concepts[relation.target].activation = max(
                        self.concepts[relation.target].activation,
                        propagated_activation
                    )
    
    def get_manifold_topology(self) -> Dict[str, Any]:
        """Get manifold topology statistics"""
        return {
            "total_concepts": len(self.concepts),
            "total_relations": len(self.relations),
            "abstraction_levels": len(self.abstraction_hierarchy),
            "connected_components": nx.number_connected_components(self.graph.to_undirected()),
            "average_degree": sum(dict(self.graph.degree()).values()) / len(self.graph.nodes()) if self.graph.nodes() else 0,
            "density": nx.density(self.graph),
            "context_windows": len(self.context_windows)
        }
    
    def visualize_manifold(self, output_file: str = "manifold.png"):
        """Visualize the semantic manifold"""
        try:
            import matplotlib.pyplot as plt
        except ImportError:
            print("matplotlib not installed. Install with: pip install synthos[viz]")
            return

        if not self.graph.nodes():
            print("No nodes to visualize")
            return
        
        plt.figure(figsize=(12, 8))
        
        # Create layout based on abstraction levels
        pos = {}
        for level in self.abstraction_hierarchy:
            y = -level  # Higher abstraction = higher y position
            for i, node_id in enumerate(self.abstraction_hierarchy[level]):
                x = i - len(self.abstraction_hierarchy[level]) / 2
                pos[node_id] = (x, y)
        
        # Draw nodes
        node_colors = []
        node_sizes = []
        for node_id in self.graph.nodes():
            if node_id in self.concepts:
                concept = self.concepts[node_id]
                node_colors.append(concept.abstraction_level)
                node_sizes.append(300 + concept.activation * 200)
            else:
                node_colors.append(0)
                node_sizes.append(300)
        
        nx.draw_networkx_nodes(
            self.graph, pos, 
            node_color=node_colors,
            node_size=node_sizes,
            cmap=plt.cm.viridis,
            alpha=0.7
        )
        
        # Draw edges with different styles for different relation types
        edge_styles = {
            RelationType.FLOWS_TO: 'solid',
            RelationType.SUBSET_OF: 'dashed',
            RelationType.EQUIVALENT_TO: 'dotted',
            RelationType.CONTAINS: 'dashdot',
            RelationType.BIDIRECTIONAL: 'solid',
            RelationType.COMBINES_WITH: 'dashed',
            RelationType.MODULATES: 'dotted'
        }
        
        for relation in self.relations:
            if relation.source in self.graph.nodes() and relation.target in self.graph.nodes():
                style = edge_styles.get(relation.relation_type, 'solid')
                nx.draw_networkx_edges(
                    self.graph, pos,
                    edgelist=[(relation.source, relation.target)],
                    style=style,
                    alpha=0.6,
                    width=relation.weight
                )
        
        # Draw labels
        labels = {node_id: node_id[:8] for node_id in self.graph.nodes()}
        nx.draw_networkx_labels(self.graph, pos, labels, font_size=8)
        
        plt.title("SYNTHOS Semantic Construct Manifold")
        plt.axis('off')
        plt.tight_layout()
        plt.savefig(output_file, dpi=300, bbox_inches='tight')
        plt.close()
        
        print(f"Manifold visualization saved to {output_file}")
    
    def query_concept_space(self, query: str, max_results: int = 10) -> List[Tuple[str, float]]:
        """Query concept space with semantic similarity"""
        query_concepts = self.extract_concepts_from_text(query)
        if not query_concepts:
            return []
        
        # Simple similarity based on shared genus and relations
        results = []
        query_genus = query_concepts[0].genus
        
        for concept_id, concept in self.concepts.items():
            if concept.genus == query_genus:
                similarity = 1.0
            elif self._are_related(query_concepts[0].concept_id, concept_id):
                similarity = 0.7
            else:
                similarity = 0.0
            
            if similarity > 0:
                results.append((concept_id, similarity))
        
        # Sort by similarity
        results.sort(key=lambda x: x[1], reverse=True)
        return results[:max_results]
    
    def _are_related(self, concept1_id: str, concept2_id: str) -> bool:
        """Check if two concepts are related"""
        for relation in self.relations:
            if (relation.source == concept1_id and relation.target == concept2_id) or \
               (relation.source == concept2_id and relation.target == concept1_id):
                return True
        return False
    
    def export_manifold(self, filename: str):
        """Export manifold to file"""
        import json
        
        export_data = {
            "concepts": {
                cid: {
                    "concept_id": c.concept_id,
                    "genus": c.genus,
                    "differentia": c.differentia,
                    "pattern": c.pattern,
                    "abstraction_level": c.abstraction_level,
                    "activation": c.activation
                }
                for cid, c in self.concepts.items()
            },
            "relations": [
                {
                    "source": r.source,
                    "target": r.target,
                    "relation_type": r.relation_type.value,
                    "weight": r.weight,
                    "bidirectional": r.bidirectional
                }
                for r in self.relations
            ],
            "topology": self.get_manifold_topology()
        }
        
        with open(filename, 'w') as f:
            json.dump(export_data, f, indent=2)
        
        print(f"Manifold exported to {filename}")

# Example usage and demonstration
if __name__ == "__main__":
    print("=== SYNTHOS SEMANTIC CONSTRUCT MANIFOLD DEMO ===")
    
    # Create manifold
    manifold = SemanticConstructManifold()
    
    # Sample text for extraction
    sample_text = """
    NeuralNetwork ⊂ DeepLearning
    DeepLearning → MachineLearning
    MachineLearning ⊂ ArtificialIntelligence
    ArtificialIntelligence ≡ CognitiveComputing
    NeuralNetwork ⊕ ConvolutionalNetwork
    ConvolutionalNetwork ⊗ ImageProcessing
    """
    
    # Extract concepts and relations
    print("\nExtracting concepts from text...")
    concepts = manifold.extract_concepts_from_text(sample_text)
    print(f"Found {len(concepts)} concepts:")
    for concept in concepts:
        print(f"  {concept}")
    
    print("\nExtracting relations from text...")
    relations = manifold.extract_relations_from_text(sample_text)
    print(f"Found {len(relations)} relations:")
    for relation in relations:
        print(f"  {relation}")
    
    # Get topology statistics
    topology = manifold.get_manifold_topology()
    print(f"\nManifold Topology:")
    for key, value in topology.items():
        print(f"  {key}: {value}")
    
    # Test concept activation
    print("\nActivating 'NeuralNetwork' concept...")
    manifold.activate_concept("NeuralNetwork", 1.0)
    
    # Show activation levels
    print("Activation levels:")
    for concept_id, concept in manifold.concepts.items():
        if concept.activation > 0:
            print(f"  {concept_id}: {concept.activation:.2f}")
    
    # Find related concepts
    related = manifold.find_related_concepts("DeepLearning")
    print(f"\nConcepts related to DeepLearning: {related}")
    
    # Query concept space
    query_results = manifold.query_concept_space("Neural")
    print(f"\nQuery results for 'Neural':")
    for concept_id, similarity in query_results:
        print(f"  {concept_id}: {similarity:.2f}")
    
    # Visualize manifold
    try:
        manifold.visualize_manifold("synthos_manifold.png")
    except Exception as e:
        print(f"Visualization failed: {e}")
    
    # Export manifold
    manifold.export_manifold("synthos_manifold.json")
    
    print("\n=== SCM DEMO COMPLETE ===")
