# ═══════════════════════════════════════════════════════════════════════════════════
# SYNTHOS OUTPUT PROJECTION SURFACE (OPS) v1.0
# Substitution chains and template geometry for output rendering
# ═══════════════════════════════════════════════════════════════════════════════════

import re
from typing import Dict, List, Tuple, Optional, Any, Callable
from dataclasses import dataclass, field
from enum import Enum
from abc import ABC, abstractmethod
import textwrap

class SubstitutionMode(Enum):
    GREEDY = "greedy"
    LAZY = "lazy"
    ATOMIC = "atomic"

class SubstitutionScope(Enum):
    GLOBAL = "global"
    FIRST = "first"
    LAST = "last"

class TemplateType(Enum):
    PARAGRAPH = "paragraph"
    LIST = "list"
    CODE = "code"
    TREE = "tree"
    DIAGRAM = "diagram"

@dataclass
class SubstitutionRule:
    """Single substitution rule in the chain"""
    find_pattern: str
    replace_template: str
    mode: SubstitutionMode
    scope: SubstitutionScope
    description: str = ""
    
    def __post_init__(self):
        # Compile regex pattern
        try:
            flags = re.MULTILINE | re.DOTALL
            if self.mode == SubstitutionMode.LAZY:
                self.find_regex = re.compile(self.find_pattern, flags | re.LOOPBACK)
            else:
                self.find_regex = re.compile(self.find_pattern, flags)
        except re.error as e:
            raise ValueError(f"Invalid regex pattern '{self.find_pattern}': {e}")
    
    def apply(self, text: str) -> str:
        """Apply substitution rule to text"""
        if self.scope == SubstitutionScope.GLOBAL:
            if self.mode == SubstitutionMode.GREEDY:
                return self.find_regex.sub(self.replace_template, text)
            elif self.mode == SubstitutionMode.LAZY:
                return self.find_regex.sub(self.replace_template, text)
            elif self.mode == SubstitutionMode.ATOMIC:
                # Atomic substitution (no backtracking)
                return self.find_regex.sub(self.replace_template, text)
        
        elif self.scope == SubstitutionScope.FIRST:
            match = self.find_regex.search(text)
            if match:
                return text[:match.start()] + match.expand(self.replace_template) + text[match.end():]
        
        elif self.scope == SubstitutionScope.LAST:
            matches = list(self.find_regex.finditer(text))
            if matches:
                last_match = matches[-1]
                return text[:last_match.start()] + last_match.expand(self.replace_template) + text[last_match.end():]
        
        return text

@dataclass
class OutputTemplate:
    """Output template with geometric rendering"""
    template_type: TemplateType
    template_pattern: str
    render_function: Callable[[Dict[str, Any]], str]
    geometric_form: str
    
    def render(self, variables: Dict[str, Any]) -> str:
        """Render template with variables"""
        return self.render_function(variables)

@dataclass
class SubstitutionChain:
    """Chain of substitution rules"""
    name: str = "default_chain"
    rules: List[SubstitutionRule] = field(default_factory=list)
    
    def add_rule(self, rule: SubstitutionRule):
        """Add rule to chain"""
        self.rules.append(rule)
    
    def apply_chain(self, input_text: str) -> str:
        """Apply entire substitution chain"""
        current_text = input_text
        
        for i, rule in enumerate(self.rules):
            current_text = rule.apply(current_text)
            
        return current_text
    
    def get_chain_summary(self) -> str:
        """Get summary of substitution chain"""
        summary = f"Chain '{self.name}' with {len(self.rules)} rules:\n"
        for i, rule in enumerate(self.rules):
            summary += f"  {i+1}. {rule.find_pattern} → {rule.replace_template} ({rule.mode.value}, {rule.scope.value})\n"
        return summary

class TemplateRenderer(ABC):
    """Abstract base class for template renderers"""
    
    @abstractmethod
    def render(self, variables: Dict[str, Any]) -> str:
        """Render template with variables"""
        pass

class ParagraphRenderer(TemplateRenderer):
    """Render paragraph block template"""
    
    def render(self, variables: Dict[str, Any]) -> str:
        topic = variables.get("TOPIC", "")
        body = variables.get("BODY", "")
        
        # Create ASCII box
        lines = [
            f"╔═ {topic} ═" + "═" * (50 - len(topic) - 5),
            f"║ {body}",
            "╚" + "═" * 54
        ]
        
        return "\n".join(lines)

class ListRenderer(TemplateRenderer):
    """Render list block template"""
    
    def render(self, variables: Dict[str, Any]) -> str:
        items = variables.get("ITEMS", [])
        if not items:
            return "└── (empty list)"
        
        lines = []
        for i, item in enumerate(items):
            if i == len(items) - 1:
                lines.append(f"└── {item}")
            else:
                lines.append(f"├── {item}")
        
        return "\n".join(lines)

class CodeRenderer(TemplateRenderer):
    """Render code block template"""
    
    def render(self, variables: Dict[str, Any]) -> str:
        lang = variables.get("LANG", "")
        code = variables.get("CODE", "")
        
        # Create ASCII code block
        lines = [
            f"┌─ {lang} " + "─" * (50 - len(lang) - 3),
            f"│ {code}",
            "└" + "─" * 54
        ]
        
        return "\n".join(lines)

class TreeRenderer(TemplateRenderer):
    """Render tree structure template"""
    
    def render(self, variables: Dict[str, Any]) -> str:
        root = variables.get("ROOT", "")
        children = variables.get("CHILDREN", [])
        
        if not children:
            return f"{root}"
        
        lines = [root]
        for i, child in enumerate(children):
            if i == len(children) - 1:
                lines.append(f"└── {child}")
            else:
                lines.append(f"├── {child}")
        
        return "\n".join(lines)

class DiagramRenderer(TemplateRenderer):
    """Render ASCII diagram template"""
    
    def render(self, variables: Dict[str, Any]) -> str:
        form = variables.get("FORM", "")
        elements = variables.get("ELEMENTS", [])
        
        if form == "attention":
            return self._render_attention_diagram(elements)
        elif form == "lattice":
            return self._render_lattice_diagram(elements)
        else:
            return f"Diagram: {form} with {len(elements)} elements"
    
    def _render_attention_diagram(self, elements: List[str]) -> str:
        """Render attention diagram"""
        return """
┌─────────────────┐
│  P1 ≡ P2        │
└─────────────────┘
        """
    
    def _render_lattice_diagram(self, elements: List[str]) -> str:
        """Render lattice diagram"""
        return """
┌─────┬─────┬─────┬─────┐
│ AA00│ AB00│ AC00│ AD00│
├─────┼─────┼─────┼─────┤
│ AA01│ AB01│ AC01│ AD01│
├─────┼─────┼─────┼─────┤
│ AA02│ AB02│ AC02│ AD02│
├─────┼─────┼─────┼─────┤
│ AA03│ AB03│ AC03│ AD03│
└─────┴─────┴─────┴─────┘
        """

class OutputProjectionSurface:
    """Main output projection surface with substitution chains and templates"""
    
    def __init__(self):
        self.substitution_chains: Dict[str, SubstitutionChain] = {}
        self.output_templates: Dict[TemplateType, OutputTemplate] = {}
        self.output_routes: Dict[str, str] = {}
        self.render_history: List[Dict[str, Any]] = []
        
        # Initialize default templates
        self._initialize_templates()
        
        # Initialize default substitution chains
        self._initialize_default_chains()
    
    def _initialize_templates(self):
        """Initialize default output templates"""
        self.output_templates[TemplateType.PARAGRAPH] = OutputTemplate(
            template_type=TemplateType.PARAGRAPH,
            template_pattern="╔═ TOPIC ════════════════════════════╗\n║ BODY ║\n╚════════════════════════════════════╝",
            render_function=ParagraphRenderer().render,
            geometric_form="◈ PARAGRAPH_BLOCK"
        )
        
        self.output_templates[TemplateType.LIST] = OutputTemplate(
            template_type=TemplateType.LIST,
            template_pattern="├── ITEM_0\n├── ITEM_1\n├── ITEM_N\n└── ITEM_LAST",
            render_function=ListRenderer().render,
            geometric_form="◈ LIST_BLOCK"
        )
        
        self.output_templates[TemplateType.CODE] = OutputTemplate(
            template_type=TemplateType.CODE,
            template_pattern="┌─ LANG ──────────────────────┐\n│ CODE │\n└───────────────────────────────────────┘",
            render_function=CodeRenderer().render,
            geometric_form="◈ CODE_BLOCK"
        )
        
        self.output_templates[TemplateType.TREE] = OutputTemplate(
            template_type=TemplateType.TREE,
            template_pattern="ROOT\n├──CHILD_MID\n└──CHILD_LAST",
            render_function=TreeRenderer().render,
            geometric_form="◈ TREE_BLOCK"
        )
        
        self.output_templates[TemplateType.DIAGRAM] = OutputTemplate(
            template_type=TemplateType.DIAGRAM,
            template_pattern="[DIAGRAM_GEOMETRY]",
            render_function=DiagramRenderer().render,
            geometric_form="◈ DIAGRAM_BLOCK"
        )
    
    def _initialize_default_chains(self):
        """Initialize default substitution chains"""
        
        # Intent → Natural Language chain
        intent_chain = SubstitutionChain("intent_to_language")
        intent_chain.add_rule(SubstitutionRule(
            find_pattern=r"EMIT\s+RESPONSE",
            replace_template="[OUTPUT]",
            mode=SubstitutionMode.GREEDY,
            scope=SubstitutionScope.GLOBAL,
            description="Mark output start"
        ))
        intent_chain.add_rule(SubstitutionRule(
            find_pattern=r"TYPE:(\w+)",
            replace_template="→mode(\\1)",
            mode=SubstitutionMode.GREEDY,
            scope=SubstitutionScope.GLOBAL,
            description="Extract mode"
        ))
        intent_chain.add_rule(SubstitutionRule(
            find_pattern=r"SUBJECT:(\w+)",
            replace_template="→topic(\\1)",
            mode=SubstitutionMode.GREEDY,
            scope=SubstitutionScope.GLOBAL,
            description="Extract topic"
        ))
        intent_chain.add_rule(SubstitutionRule(
            find_pattern=r"DEPTH:(\d+)",
            replace_template="→depth(\\1)",
            mode=SubstitutionMode.GREEDY,
            scope=SubstitutionScope.GLOBAL,
            description="Extract depth"
        ))
        intent_chain.add_rule(SubstitutionRule(
            find_pattern=r"\[OUTPUT\]\s*→mode\((\w+)\)\s*→topic\((\w+)\)\s*→depth\((\d+)\)",
            replace_template="RENDER_\\1(\\2, verbosity=\\3)",
            mode=SubstitutionMode.GREEDY,
            scope=SubstitutionScope.GLOBAL,
            description="Compose final render call"
        ))
        
        self.substitution_chains["intent_to_language"] = intent_chain
        
        # Code formatting chain
        code_chain = SubstitutionChain("code_formatting")
        code_chain.add_rule(SubstitutionRule(
            find_pattern=r"```(\w+)",
            replace_template="┌─ \\1 ──────────────────────┐\n│",
            mode=SubstitutionMode.GREEDY,
            scope=SubstitutionScope.GLOBAL,
            description="Start code block"
        ))
        code_chain.add_rule(SubstitutionRule(
            find_pattern=r"```",
            replace_template="\n└───────────────────────────────────────┘",
            mode=SubstitutionMode.GREEDY,
            scope=SubstitutionScope.GLOBAL,
            description="End code block"
        ))
        
        self.substitution_chains["code_formatting"] = code_chain
    
    def add_substitution_chain(self, chain: SubstitutionChain):
        """Add a substitution chain"""
        self.substitution_chains[chain.name] = chain
    
    def add_output_route(self, route_name: str, template_type: TemplateType):
        """Add output routing rule"""
        self.output_routes[route_name] = template_type.value
    
    def process_input(self, input_text: str, chain_name: str = "intent_to_language") -> str:
        """Process input through substitution chain"""
        if chain_name not in self.substitution_chains:
            raise ValueError(f"Unknown substitution chain: {chain_name}")
        
        chain = self.substitution_chains[chain_name]
        output = chain.apply_chain(input_text)
        
        # Record processing
        self.render_history.append({
            "input": input_text,
            "output": output,
            "chain": chain_name,
            "timestamp": len(self.render_history)
        })
        
        return output
    
    def render_template(self, template_type: TemplateType, variables: Dict[str, Any]) -> str:
        """Render output template"""
        if template_type not in self.output_templates:
            raise ValueError(f"Unknown template type: {template_type}")
        
        template = self.output_templates[template_type]
        output = template.render(variables)
        
        # Record rendering
        self.render_history.append({
            "template": template_type.value,
            "variables": variables,
            "output": output,
            "timestamp": len(self.render_history)
        })
        
        return output
    
    def route_output(self, route_name: str, variables: Dict[str, Any]) -> str:
        """Route output through template"""
        if route_name not in self.output_routes:
            raise ValueError(f"Unknown output route: {route_name}")
        
        template_type_str = self.output_routes[route_name]
        template_type = TemplateType(template_type_str)
        
        return self.render_template(template_type, variables)
    
    def parse_render_command(self, command: str) -> Tuple[str, Dict[str, Any]]:
        """Parse render command like RENDER_EXPLAIN(neural_nets, verbosity=3)"""
        # Parse RENDER_FUNCTION(args) pattern
        match = re.match(r'RENDER_(\w+)\(([^)]+)\)', command)
        if not match:
            return command, {}
        
        function_name = match.group(1)
        args_str = match.group(2)
        
        # Parse arguments
        variables = {}
        if args_str:
            # Simple argument parsing
            args = args_str.split(',')
            for arg in args:
                arg = arg.strip()
                if '=' in arg:
                    key, value = arg.split('=', 1)
                    variables[key.strip()] = value.strip()
                else:
                    variables['topic'] = arg.strip()
        
        return function_name, variables
    
    def execute_render_command(self, command: str) -> str:
        """Execute a render command"""
        function_name, variables = self.parse_render_command(command)
        
        # Map function names to template types
        template_mapping = {
            "EXPLAIN": TemplateType.PARAGRAPH,
            "LIST": TemplateType.LIST,
            "CODE": TemplateType.CODE,
            "TREE": TemplateType.TREE,
            "DIAGRAM": TemplateType.DIAGRAM
        }
        
        if function_name in template_mapping:
            template_type = template_mapping[function_name]
            return self.render_template(template_type, variables)
        else:
            return f"Unknown render function: {function_name}"
    
    def get_processing_statistics(self) -> Dict[str, Any]:
        """Get processing statistics"""
        if not self.render_history:
            return {"total_operations": 0}
        
        chain_usage = {}
        template_usage = {}
        
        for entry in self.render_history:
            if "chain" in entry:
                chain_usage[entry["chain"]] = chain_usage.get(entry["chain"], 0) + 1
            if "template" in entry:
                template_usage[entry["template"]] = template_usage.get(entry["template"], 0) + 1
        
        return {
            "total_operations": len(self.render_history),
            "chain_usage": chain_usage,
            "template_usage": template_usage,
            "available_chains": list(self.substitution_chains.keys()),
            "available_templates": [t.value for t in self.output_templates.keys()],
            "available_routes": list(self.output_routes.keys())
        }
    
    def export_configuration(self, filename: str):
        """Export configuration to file"""
        import json
        
        export_data = {
            "substitution_chains": {
                name: {
                    "name": chain.name,
                    "rules": [
                        {
                            "find_pattern": rule.find_pattern,
                            "replace_template": rule.replace_template,
                            "mode": rule.mode.value,
                            "scope": rule.scope.value,
                            "description": rule.description
                        }
                        for rule in chain.rules
                    ]
                }
                for name, chain in self.substitution_chains.items()
            },
            "output_routes": self.output_routes,
            "statistics": self.get_processing_statistics()
        }
        
        with open(filename, 'w') as f:
            json.dump(export_data, f, indent=2)
        
        print(f"Configuration exported to {filename}")
    
    def import_configuration(self, filename: str):
        """Import configuration from file"""
        import json
        
        with open(filename, 'r') as f:
            import_data = json.load(f)
        
        # Import substitution chains
        for name, chain_data in import_data.get("substitution_chains", {}).items():
            chain = SubstitutionChain(chain_data["name"])
            for rule_data in chain_data["rules"]:
                rule = SubstitutionRule(
                    find_pattern=rule_data["find_pattern"],
                    replace_template=rule_data["replace_template"],
                    mode=SubstitutionMode(rule_data["mode"]),
                    scope=SubstitutionScope(rule_data["scope"]),
                    description=rule_data.get("description", "")
                )
                chain.add_rule(rule)
            self.substitution_chains[name] = chain
        
        # Import output routes
        self.output_routes.update(import_data.get("output_routes", {}))
        
        print(f"Configuration imported from {filename}")

# Example usage and demonstration
if __name__ == "__main__":
    print("=== SYNTHOS OUTPUT PROJECTION SURFACE DEMO ===")
    
    # Create output projection surface
    ops = OutputProjectionSurface()
    
    # Add output routes
    ops.add_output_route("EXPLAIN", TemplateType.PARAGRAPH)
    ops.add_output_route("LIST", TemplateType.LIST)
    ops.add_output_route("CODE", TemplateType.CODE)
    ops.add_output_route("DIAGRAM", TemplateType.DIAGRAM)
    
    # Test substitution chain
    test_input = "EMIT RESPONSE TYPE:EXPLAIN SUBJECT:neural_nets DEPTH:3"
    print(f"\nProcessing input: '{test_input}'")
    
    processed = ops.process_input(test_input, "intent_to_language")
    print(f"Processed output: '{processed}'")
    
    # Execute render command
    render_output = ops.execute_render_command(processed)
    print(f"Rendered output:\n{render_output}")
    
    # Test template rendering directly
    print(f"\nTesting template rendering:")
    
    # Paragraph template
    paragraph_vars = {"TOPIC": "Neural Networks", "BODY": "Neural networks are computational models inspired by biological neural systems."}
    paragraph_output = ops.render_template(TemplateType.PARAGRAPH, paragraph_vars)
    print(f"\nParagraph Template:\n{paragraph_output}")
    
    # List template
    list_vars = {"ITEMS": ["Input layer", "Hidden layers", "Output layer"]}
    list_output = ops.render_template(TemplateType.LIST, list_vars)
    print(f"\nList Template:\n{list_output}")
    
    # Code template
    code_vars = {"LANG": "Python", "CODE": "def neural_network():\n    pass"}
    code_output = ops.render_template(TemplateType.CODE, code_vars)
    print(f"\nCode Template:\n{code_output}")
    
    # Tree template
    tree_vars = {"ROOT": "Network", "CHILDREN": ["Input", "Hidden", "Output"]}
    tree_output = ops.render_template(TemplateType.TREE, tree_vars)
    print(f"\nTree Template:\n{tree_output}")
    
    # Diagram template
    diagram_vars = {"FORM": "attention", "ELEMENTS": ["P1", "P2"]}
    diagram_output = ops.render_template(TemplateType.DIAGRAM, diagram_vars)
    print(f"\nDiagram Template:\n{diagram_output}")
    
    # Test output routing
    print(f"\nTesting output routing:")
    routed_output = ops.route_output("EXPLAIN", {"TOPIC": "AI", "BODY": "Artificial Intelligence is transforming technology."})
    print(f"Routed output:\n{routed_output}")
    
    # Show chain summary
    print(f"\nChain Summary:")
    intent_chain = ops.substitution_chains["intent_to_language"]
    print(intent_chain.get_chain_summary())
    
    # Get statistics
    stats = ops.get_processing_statistics()
    print(f"\nProcessing Statistics:")
    for key, value in stats.items():
        print(f"  {key}: {value}")
    
    # Export configuration
    ops.export_configuration("synthos_ops_config.json")
    
    # Test custom substitution chain
    custom_chain = SubstitutionChain("custom_chain")
    custom_chain.add_rule(SubstitutionRule(
        find_pattern=r"ERROR:(\w+)",
        replace_template="[EXCEPTION: \\1]",
        mode=SubstitutionMode.GREEDY,
        scope=SubstitutionScope.GLOBAL,
        description="Format error messages"
    ))
    
    ops.add_substitution_chain(custom_chain)
    
    # Test custom chain
    custom_input = "ERROR:InvalidSyntax ERROR:MissingToken"
    custom_output = ops.process_input(custom_input, "custom_chain")
    print(f"\nCustom chain processing:")
    print(f"Input: {custom_input}")
    print(f"Output: {custom_output}")
    
    print("\n=== OPS DEMO COMPLETE ===")
