"""
SYNTHOS Command-Line Interface
================================

System-wide entry point installed as ``synthos``.  Provides subcommands for
every major capability:

    synthos              Launch the interactive TUI (default)
    synthos chat         Headless REPL (no Textual, plain Rich output)
    synthos encrypt      Encrypt stdin / file with STC cipher
    synthos decrypt      Decrypt STC envelope
    synthos eval         Run model benchmark
    synthos status       Print pipeline status
    synthos config       Show / edit configuration
    synthos version      Print version

All subcommands honour ``--verbose`` / ``-v`` (stackable up to ``-vv``).
"""

from __future__ import annotations

import os
import sys
from pathlib import Path

import click
from rich.console import Console
from rich.panel import Panel
from rich.table import Table
from rich.text import Text

from synthos import __version__
from synthos.utils.config import load_config, save_config, SynthosConfig
from synthos.utils.log import VerboseLevel, get_logger

console = Console()


def _safe_console() -> Console:
    """Return a Console that works in any environment, including dumb terminals."""
    try:
        if sys.stdout.isatty():
            return Console()
        else:
            # Non-interactive: disable color/markup so output is plain text
            return Console(force_terminal=False, no_color=True, highlight=False)
    except Exception:
        return Console(force_terminal=False, no_color=True, highlight=False)


def _can_run_tui() -> bool:
    """Return True if the terminal supports Textual's full TUI."""
    # Not a real TTY (pipe, cron, CI, etc.)
    if not sys.stdin.isatty() or not sys.stdout.isatty():
        return False
    # $TERM is dumb or unset
    term = os.environ.get("TERM", "")
    if term in ("", "dumb", "unknown"):
        return False
    # Textual not importable
    try:
        import textual  # noqa: F401
        return True
    except ImportError:
        return False


# ── Verbose option helper ──────────────────────────────────────────────────────

def _vlevel(ctx: click.Context) -> VerboseLevel:
    v = ctx.obj.get("verbose", 0)
    return VerboseLevel(min(v, 2))


# ── Root group ─────────────────────────────────────────────────────────────────

@click.group(invoke_without_command=True)
@click.option("-v", "--verbose", count=True, help="Increase verbosity (-v normal, -vv verbose).")
@click.option("--cli", "force_cli", is_flag=True, default=False,
              help="Force plain-text REPL mode (skip Textual TUI).")
@click.option("--config", "config_path", type=click.Path(exists=False), default=None,
              help="Path to config YAML (default ~/.config/synthos/config.yaml).")
@click.version_option(__version__, prog_name="synthos")
@click.pass_context
def main(ctx: click.Context, verbose: int, force_cli: bool, config_path: str | None):
    """
    SYNTHOS — Syntactic Topological Hierarchy for Organized Symbol-based Systems.

    A regex/syntax-driven AI architecture with built-in encryption,
    model evaluation, and a rich interactive TUI.

    Run without a subcommand to launch the TUI (falls back to plain REPL
    if the terminal does not support it).  Use --cli to force REPL mode.
    """
    ctx.ensure_object(dict)
    ctx.obj["verbose"] = verbose
    ctx.obj["force_cli"] = force_cli
    cfg_path = Path(config_path) if config_path else None
    ctx.obj["config"] = load_config(cfg_path)

    if ctx.invoked_subcommand is None:
        if force_cli or not _can_run_tui():
            ctx.invoke(chat)
        else:
            try:
                ctx.invoke(tui)
            except Exception:
                # TUI failed to start — fall back to REPL
                ctx.invoke(chat)


# ── synthos tui ────────────────────────────────────────────────────────────────

@main.command()
@click.option("-w", "--workspace", default=None, help="Workspace directory.")
@click.pass_context
def tui(ctx: click.Context, workspace: str):
    """Launch the interactive SYNTHOS TUI (default)."""
    from synthos.tui.app import run_tui
    cfg: SynthosConfig = ctx.obj["config"]
    run_tui(config=cfg, verbose=_vlevel(ctx), workspace=workspace)


# ── synthos chat ───────────────────────────────────────────────────────────────

@main.command()
@click.option("-w", "--workspace", default=None, help="Workspace directory.")
@click.pass_context
def chat(ctx: click.Context, workspace: str):
    """Start a headless REPL with full SynthoLM (no Textual dependency)."""
    from synthos.lm.engine import SynthoLM, GenerationConfig

    cfg: SynthosConfig = ctx.obj["config"]
    vlevel = _vlevel(ctx)

    # Use a safe console that degrades gracefully
    _con = _safe_console()

    lm = SynthoLM(config=cfg, verbose=vlevel, workspace=workspace)
    lm._auto_approve = True

    # Configurable settings (same as TUI)
    max_words = 4000
    max_search_queries = 12
    max_sources = 8

    _con.print(Panel.fit(
        "[bold cyan]SYNTHOS[/bold cyan] v{ver}  ·  7 layers  ·  Deep Research  ·  STC cipher\n"
        "Type [bold]/help[/bold] for commands, [bold]Ctrl+D[/bold] or [bold]/quit[/bold] to exit.".format(ver=__version__),
        border_style="cyan",
    ))

    while True:
        try:
            user_input = _con.input("[bold green]synthos>[/bold green] ")
        except (EOFError, KeyboardInterrupt):
            _con.print("\n[dim]Goodbye.[/dim]")
            break

        text = user_input.strip()
        if not text:
            continue

        # ── Slash commands ────────────────────────────────────
        if text.lower() in ("/quit", "/exit", "/q"):
            _con.print("[dim]Goodbye.[/dim]")
            break
        elif text.lower() == "/help":
            _con.print(
                "\n[bold cyan]Commands:[/bold cyan]\n"
                "  [green]/help[/green]              Show this help\n"
                "  [green]/quit[/green]              Exit SYNTHOS\n"
                "  [green]/set words <N>[/green]     Max response words (current: {w})\n"
                "  [green]/set queries <N>[/green]   Search queries count (current: {q})\n"
                "  [green]/set sources <N>[/green]   Max sources shown (current: {s})\n"
                "  [green]/verbose[/green]           Cycle verbosity\n"
                "  [green]/status[/green]            Show pipeline status\n"
                .format(w=max_words, q=max_search_queries, s=max_sources)
            )
            continue
        elif text.lower() == "/verbose":
            vlevel = VerboseLevel((vlevel.value + 1) % 3)
            _con.print(f"[yellow]Verbosity → {vlevel.name}[/yellow]")
            continue
        elif text.lower() == "/status":
            from synthos.assistant import SynthosAssistant
            _a = SynthosAssistant(config=cfg, verbose=vlevel)
            s = _a.get_status()
            for k, v in s.items():
                _con.print(f"  [cyan]{k}[/cyan]: {v}")
            continue
        elif text.lower().startswith("/set "):
            parts = text.split()
            if len(parts) >= 3:
                param, val = parts[1].lower(), parts[2]
                try:
                    val_i = int(val)
                except ValueError:
                    _con.print("[red]Value must be an integer.[/red]")
                    continue
                if param == "words":
                    max_words = max(50, min(50000, val_i))
                    _con.print(f"[green]✓ Max words → {max_words}[/green]")
                elif param == "queries":
                    max_search_queries = max(1, min(20, val_i))
                    _con.print(f"[green]✓ Search queries → {max_search_queries}[/green]")
                elif param == "sources":
                    max_sources = max(1, min(20, val_i))
                    _con.print(f"[green]✓ Max sources → {max_sources}[/green]")
                else:
                    _con.print(f"[red]Unknown setting: {param}[/red]")
            continue

        # ── Generate response ─────────────────────────────────
        try:
            gen_cfg = GenerationConfig(
                max_words=max_words,
                max_search_queries=max_search_queries,
                max_sources=max_sources,
            )
            result = lm.generate(text, config=gen_cfg)
            _con.print(f"\n[cyan]{result.text}[/cyan]")
            if vlevel >= VerboseLevel.VERBOSE:
                _con.print(f"[dim]{result.mode} · {result.tokens_generated} tokens · {result.latency_ms:.0f}ms[/dim]")
            _con.print()
        except Exception as exc:
            _con.print(f"[red]Error: {exc}[/red]\n")


# ── synthos encrypt ────────────────────────────────────────────────────────────

@main.command()
@click.argument("plaintext", required=False)
@click.option("-f", "--file", "infile", type=click.Path(exists=True), help="File to encrypt.")
@click.option("-k", "--key", "key_hex", default=None, help="Hex-encoded 256-bit key (generated if omitted).")
@click.option("-m", "--mode", type=click.Choice(["ecb", "cbc", "ctr"]), default="cbc", help="Cipher mode.")
@click.option("-o", "--output", "outfile", type=click.Path(), default=None, help="Write ciphertext to file.")
@click.pass_context
def encrypt(ctx: click.Context, plaintext: str | None, infile: str | None, key_hex: str | None,
            mode: str, outfile: str | None):
    """Encrypt text or a file with the Symbolic Topology Cipher."""
    from synthos.crypto.stc import SymbolicTopologyCipher, STCKey, STCMode

    vlevel = _vlevel(ctx)

    # Resolve plaintext
    if infile:
        data = Path(infile).read_bytes()
    elif plaintext:
        data = plaintext.encode()
    elif not sys.stdin.isatty():
        data = sys.stdin.buffer.read()
    else:
        console.print("[red]Provide plaintext as argument, --file, or pipe via stdin.[/red]")
        raise SystemExit(1)

    # Resolve key
    if key_hex:
        key = STCKey.from_hex(key_hex)
    else:
        key = STCKey.generate()
        console.print(f"[yellow]Generated key:[/yellow] {key.hex()}")

    cipher = SymbolicTopologyCipher(key, STCMode(mode), verbose=(vlevel >= VerboseLevel.VERBOSE))
    ct = cipher.encrypt(data)

    if outfile:
        Path(outfile).write_bytes(ct)
        console.print(f"[green]Ciphertext written to {outfile} ({len(ct)} bytes)[/green]")
    else:
        console.print(f"[green]{ct.hex()}[/green]")

    if vlevel >= VerboseLevel.VERBOSE and cipher.trace:
        console.print(Panel("\n".join(cipher.trace[:30]), title="STC Trace", border_style="green"))


# ── synthos decrypt ────────────────────────────────────────────────────────────

@main.command()
@click.argument("ciphertext_hex", required=False)
@click.option("-f", "--file", "infile", type=click.Path(exists=True), help="File to decrypt.")
@click.option("-k", "--key", "key_hex", required=True, help="Hex-encoded 256-bit key.")
@click.option("-m", "--mode", type=click.Choice(["ecb", "cbc", "ctr"]), default="cbc", help="Cipher mode.")
@click.option("-o", "--output", "outfile", type=click.Path(), default=None, help="Write plaintext to file.")
@click.pass_context
def decrypt(ctx: click.Context, ciphertext_hex: str | None, infile: str | None, key_hex: str,
            mode: str, outfile: str | None):
    """Decrypt an STC envelope."""
    from synthos.crypto.stc import SymbolicTopologyCipher, STCKey, STCMode

    vlevel = _vlevel(ctx)
    key = STCKey.from_hex(key_hex)

    if infile:
        ct = Path(infile).read_bytes()
    elif ciphertext_hex:
        ct = bytes.fromhex(ciphertext_hex.replace('\n', '').replace(' ', ''))
    elif not sys.stdin.isatty():
        ct = bytes.fromhex(sys.stdin.read().replace('\n', '').replace(' ', ''))
    else:
        console.print("[red]Provide ciphertext as hex argument, --file, or pipe via stdin.[/red]")
        raise SystemExit(1)

    cipher = SymbolicTopologyCipher(key, STCMode(mode), verbose=(vlevel >= VerboseLevel.VERBOSE))
    try:
        pt = cipher.decrypt(ct)
    except ValueError as exc:
        console.print(f"[red]Decryption failed: {exc}[/red]")
        raise SystemExit(1)

    if outfile:
        Path(outfile).write_bytes(pt)
        console.print(f"[green]Plaintext written to {outfile} ({len(pt)} bytes)[/green]")
    else:
        console.print(pt.decode(errors="replace"))


# ── synthos eval ───────────────────────────────────────────────────────────────

@main.command(name="eval")
@click.option("-s", "--suite", multiple=True, default=None,
              help="Eval suite name (repeatable). Default: all built-in suites.")
@click.option("--json", "as_json", is_flag=True, help="Output results as JSON.")
@click.pass_context
def eval_cmd(ctx: click.Context, suite: tuple[str, ...], as_json: bool):
    """Run model benchmark against configured endpoints."""
    from synthos.eval.bench import ModelBenchmark, ModelCard, EvalSuite, Provider

    cfg: SynthosConfig = ctx.obj["config"]
    vlevel = _vlevel(ctx)

    cards = [
        ModelCard(
            name=ep.name, provider=Provider(ep.provider), base_url=ep.base_url,
            model_id=ep.model_id, ctx_len=ep.ctx_len, params_b=ep.params_b,
            quant=ep.quant, cost_per_mtok=ep.cost_per_mtok,
        )
        for ep in cfg.models
    ]

    suites = [EvalSuite(s) for s in suite] if suite else None
    bench = ModelBenchmark(cards, suites, verbose=(vlevel >= VerboseLevel.VERBOSE))

    console.print("[cyan]Running evaluation …[/cyan]")
    try:
        results = bench.run()
    except Exception as exc:
        console.print(f"[red]Eval failed: {exc}[/red]")
        console.print("[dim]Ensure an inference server is running at the configured endpoint.[/dim]")
        raise SystemExit(1)

    if as_json:
        console.print(ModelBenchmark.to_json(results))
    else:
        console.print(ModelBenchmark.leaderboard(results))


# ── synthos status ─────────────────────────────────────────────────────────────

@main.command()
@click.pass_context
def status(ctx: click.Context):
    """Print SYNTHOS pipeline status."""
    from synthos.assistant import SynthosAssistant

    cfg: SynthosConfig = ctx.obj["config"]
    assistant = SynthosAssistant(config=cfg, verbose=_vlevel(ctx))
    s = assistant.get_status()

    table = Table(title="SYNTHOS System Status", show_header=True, header_style="bold magenta")
    table.add_column("Subsystem", style="cyan", width=28)
    table.add_column("Value", style="green")

    for k, v in s.items():
        table.add_row(k, str(v))

    console.print(table)


# ── synthos config ─────────────────────────────────────────────────────────────

@main.command()
@click.option("--edit", is_flag=True, help="Open config in $EDITOR.")
@click.pass_context
def config(ctx: click.Context, edit: bool):
    """Show or edit SYNTHOS configuration."""
    import yaml, os, subprocess
    from dataclasses import asdict
    from synthos.utils.config import DEFAULT_CONFIG_FILE

    cfg: SynthosConfig = ctx.obj["config"]

    if edit:
        editor = os.environ.get("EDITOR", "vi")
        save_config(cfg)
        subprocess.call([editor, str(DEFAULT_CONFIG_FILE)])
        console.print(f"[green]Config saved to {DEFAULT_CONFIG_FILE}[/green]")
    else:
        console.print(Panel(
            yaml.dump(asdict(cfg), default_flow_style=False, sort_keys=False),
            title=str(DEFAULT_CONFIG_FILE),
            border_style="cyan",
        ))


# ── synthos do ─────────────────────────────────────────────────────────────────

@main.command("do")
@click.argument("task", nargs=-1, required=True)
@click.option("-w", "--workspace", default=None, help="Workspace directory (default: ~/synthos_workspace).")
@click.option("-y", "--yes", is_flag=True, default=False, help="Auto-approve all actions.")
@click.pass_context
def do_task(ctx: click.Context, task: tuple, workspace: str, yes: bool):
    """Execute a plain-English task (create files, dirs, scaffold projects, etc.)."""
    import time as _time
    from synthos.lm.engine import SynthoLM
    from synthos.utils.log import VerboseLevel
    from synthos.tools.planner import TaskPlanner

    verb = VerboseLevel(min(ctx.obj.get("verbose", 1), 2))
    lm = SynthoLM(verbose=verb, workspace=workspace)

    request = " ".join(task)
    console.print(f"[cyan]Task:[/cyan] {request}\n")

    t0 = _time.perf_counter()

    # 1. Plan
    plan = lm.planner.plan(request)
    has_actions = any(s.action is not None for s in plan.steps)

    if not has_actions:
        # Pure reasoning — fall through to LM
        result = lm.generate(request)
        console.print(Panel(result.text, title="SYNTHOS Response", border_style="cyan"))
        console.print(f"\n[dim]{result.mode} · {result.tokens_generated} tokens · {result.latency_ms:.0f}ms[/dim]")
        return

    # 2. Show plan and ask for approval
    console.print(Panel("[bold]Execution Plan[/bold]", border_style="yellow"))

    groups = plan.parallel_groups
    for gid in sorted(groups.keys()):
        group_steps = groups[gid]
        if len(group_steps) > 1:
            console.print(f"  [yellow]⟐ Parallel Group {gid + 1}[/yellow] — {len(group_steps)} steps run simultaneously")
        for step in group_steps:
            dep_str = ""
            if step.depends_on:
                dep_str = f" [dim](after step {', '.join(str(d) for d in step.depends_on)})[/dim]"
            marker = "[red]⚠[/red]" if step.is_destructive else "[green]•[/green]"
            console.print(f"    {marker} Step {step.id}: {step.description}{dep_str}")
            if verb.value >= 2:
                console.print(f"      [dim]{step.reasoning}[/dim]")

    total = len(plan.steps)
    parallel_groups = sum(1 for g in groups.values() if len(g) > 1)
    destructive = sum(1 for s in plan.steps if s.is_destructive)
    console.print(f"\n  [dim]{total} step(s), {parallel_groups} parallel group(s)[/dim]")
    if destructive:
        console.print(f"  [red]⚠ {destructive} destructive step(s) need your approval[/red]")

    # 3. Approval
    if yes:
        approved = True
        console.print("\n  [yellow]Auto-approved (--yes flag)[/yellow]")
    else:
        console.print()
        approved = click.confirm("  Shall I proceed?", default=True)

    if not approved:
        console.print("\n[yellow]Cancelled. No actions were taken.[/yellow]")
        return

    # 4. Execute with progress
    console.print("\n[bold green]Executing...[/bold green]\n")
    lm.planner.approve(plan)

    def _progress(step, msg):
        console.print(f"  [dim]→ Step {step.id}: {msg}[/dim]")

    lm.planner.set_progress_callback(_progress)
    response = lm.planner.execute(plan)

    wall_ms = (_time.perf_counter() - t0) * 1000
    console.print(Panel(response, title="SYNTHOS Execution Result", border_style="green"))
    console.print(f"\n[dim]tool_action · {total} steps · {wall_ms:.0f}ms[/dim]")


# ── synthos serve ──────────────────────────────────────────────────────────────

@main.command()
@click.option("-p", "--port", default=8100, help="Port to listen on.")
@click.option("--host", default="0.0.0.0", help="Host to bind to.")
@click.pass_context
def serve(ctx: click.Context, port: int, host: str):
    """Start the SYNTHOS API server (FastAPI + SynthoLM)."""
    from synthos.server import run_server
    console.print(f"[cyan]Starting SYNTHOS API on {host}:{port} …[/cyan]")
    run_server(host=host, port=port)


# ── synthos version ────────────────────────────────────────────────────────────

@main.command()
def version():
    """Print SYNTHOS version."""
    console.print(f"synthos {__version__}")
