"""
SYNTHOS AI Assistant Engine
============================

The central orchestrator that wires all six cognitive layers, the encryption
engine, and the model-evaluation subsystem into a single conversational
assistant.

Signal flow::

    user_input
        │
        ├─► LPE  (tokenise into regex primitives)
        ├─► GPL  (route through parse lattice)
        ├─► SCM  (bind semantic concepts & relations)
        ├─► TAM  (multi-head attention intersections)
        ├─► RGE  (grammar parse, recursive expansion)
        ├─► SCF  (crystallise state, update memory)
        └─► OPS  (substitution chain → structured output)

Special prefixes are intercepted before the pipeline:
    /encrypt <text>      — encrypt with STC
    /decrypt <hex>       — decrypt STC envelope
    /eval [model …]      — run model benchmark
    /status              — show live system status
    /layers              — layer diagnostics
    /config              — print active config
    /verbose [0|1|2]     — change verbosity at runtime
"""

from __future__ import annotations

import re
import time
from dataclasses import dataclass, field
from typing import Any, Dict, List, Optional

from synthos.layers.lpe import PrimitiveRegistry, PrimitiveType
from synthos.layers.gpl import GeometricParseLattice
from synthos.layers.scm import SemanticConstructManifold, RelationType
from synthos.layers.tam import TopologicalAttentionMesh
from synthos.layers.rge import RecursiveGrammarEngine
from synthos.layers.scf import StateCrystallizationField, LayerType
from synthos.layers.ops import OutputProjectionSurface, TemplateType

from synthos.crypto.stc import SymbolicTopologyCipher, STCKey, STCMode
from synthos.utils.config import SynthosConfig, load_config
from synthos.utils.log import get_logger, VerboseLevel


@dataclass
class PipelineTrace:
    """Captures per-layer diagnostics for one request."""
    input_text: str = ""
    wall_ms: float = 0.0
    layers: Dict[str, Dict[str, Any]] = field(default_factory=dict)
    final_output: str = ""


class SynthosAssistant:
    """
    Full SYNTHOS AI assistant.

    Parameters
    ----------
    config : SynthosConfig | None
        System configuration; loaded from disk when *None*.
    verbose : VerboseLevel | None
        Override config verbosity for this session.
    """

    BANNER = r"""
╔══════════════════════════════════════════════════════════════════╗
║   ____  __  __ _  _ _____ _  _  ___  ____                      ║
║  / ___) \ \/ /| \| |_   _| || |/ _ \/ ___|                     ║
║  \___ \  \  / |  \ | | | | __ | | | \___ \                     ║
║  (____/  |  | | |\  | | | |  ||_| |_|(____/                    ║
║          |__|_|_| \_| |_| |_||_|\___/                          ║
║                                                                  ║
║  Syntactic Topological Hierarchy for Organized Symbol Systems    ║
║  v3.0  ·  7 layers  ·  STC cipher  ·  model eval                ║
╚══════════════════════════════════════════════════════════════════╝
"""

    def __init__(
        self,
        config: Optional[SynthosConfig] = None,
        verbose: Optional[VerboseLevel] = None,
    ):
        self.config = config or load_config()
        self.vlevel = verbose if verbose is not None else VerboseLevel(self.config.verbose)
        self.log = get_logger("synthos.assistant", self.vlevel)

        self.log.info("[layer]Booting SYNTHOS cognitive pipeline …[/layer]")

        # ── Layers ─────────────────────────────────────────────────────────
        self.lpe = PrimitiveRegistry()
        self.gpl = GeometricParseLattice(
            rows=self.config.lattice_rows,
            cols=self.config.lattice_cols,
        )
        self.scm = SemanticConstructManifold()
        self.tam = TopologicalAttentionMesh(num_heads=self.config.attention_heads)
        self.rge = RecursiveGrammarEngine()
        self.scf = StateCrystallizationField(
            tensor_shape=(self.config.lattice_rows, self.config.lattice_cols),
        )
        self.ops = OutputProjectionSurface()

        # Output routes
        self.ops.add_output_route("EXPLAIN", TemplateType.PARAGRAPH)
        self.ops.add_output_route("LIST", TemplateType.LIST)
        self.ops.add_output_route("CODE", TemplateType.CODE)
        self.ops.add_output_route("DIAGRAM", TemplateType.DIAGRAM)

        # ── Crypto ─────────────────────────────────────────────────────────
        self._stc_key: Optional[STCKey] = None

        # ── Conversation history (episodic memory) ─────────────────────────
        self.history: List[Dict[str, str]] = []

        self.log.info("[layer]Pipeline online  (verbose=%s)[/layer]", self.vlevel.name)

    # ── Public API ─────────────────────────────────────────────────────────

    def process(self, user_input: str) -> str:
        """
        Process one user message and return the assistant response.

        Intercepts ``/``-commands; everything else flows through the
        7-layer pipeline.
        """
        user_input = user_input.strip()
        if not user_input:
            return ""

        self.history.append({"role": "user", "content": user_input})

        # ── Command dispatch ───────────────────────────────────────────────
        if user_input.startswith("/"):
            response = self._handle_command(user_input)
        else:
            response = self._pipeline(user_input)

        self.history.append({"role": "assistant", "content": response})
        return response

    def get_status(self) -> Dict[str, Any]:
        """Return a summary dict of every subsystem."""
        return {
            "primitives": len(self.lpe.primitives),
            "lattice_cells": len(self.gpl.cells),
            "concepts": len(self.scm.concepts),
            "relations": len(self.scm.relations),
            "attention_heads": len(self.tam.attention_heads),
            "grammar_rules": len(self.rge.production_rules),
            "coherence": self.scf.compute_coherence(),
            "memory_st": len(self.scf.memory_lattice.short_term_buffer),
            "memory_lt": len(self.scf.memory_lattice.long_term_registers),
            "memory_ep": len(self.scf.memory_lattice.episodic_stack),
            "history_turns": len(self.history),
            "verbose": self.vlevel.name,
        }

    # ── 7-layer pipeline ──────────────────────────────────────────────────

    def _pipeline(self, text: str) -> str:
        t0 = time.perf_counter()
        trace = PipelineTrace(input_text=text)

        # L0 — Lexical Primitive Engine
        self.log.info("[layer]L0  LPE[/layer]  extracting primitives")
        lpe_out = self._layer_lpe(text)
        trace.layers["lpe"] = lpe_out

        # L1 — Geometric Parse Lattice
        self.log.info("[layer]L1  GPL[/layer]  lattice traversal")
        gpl_out = self._layer_gpl(text)
        trace.layers["gpl"] = gpl_out

        # L2 — Semantic Construct Manifold
        self.log.info("[layer]L2  SCM[/layer]  concept binding")
        scm_out = self._layer_scm(text)
        trace.layers["scm"] = scm_out

        # L3 — Topological Attention Mesh
        self.log.info("[layer]L3  TAM[/layer]  attention intersections")
        tam_out = self._layer_tam(text)
        trace.layers["tam"] = tam_out

        # L4 — Recursive Grammar Engine
        self.log.info("[layer]L4  RGE[/layer]  grammar parse")
        rge_out = self._layer_rge(text)
        trace.layers["rge"] = rge_out

        # L5 — State Crystallization Field
        self.log.info("[layer]L5  SCF[/layer]  state crystallisation")
        scf_out = self._layer_scf(text, tam_out)
        trace.layers["scf"] = scf_out

        # L6 — Output Projection Surface
        self.log.info("[layer]L6  OPS[/layer]  output projection")
        output = self._layer_ops(text, scf_out)

        trace.wall_ms = (time.perf_counter() - t0) * 1000
        trace.final_output = output

        if self.vlevel >= VerboseLevel.VERBOSE:
            self.log.debug("[dim]Pipeline completed in %.1f ms[/dim]", trace.wall_ms)

        # Store trace as episodic memory
        self.scf.memory_lattice.add_episode({
            "input": text,
            "output": output,
            "wall_ms": trace.wall_ms,
        })

        return output

    # ── Per-layer implementations ──────────────────────────────────────────

    def _layer_lpe(self, text: str) -> Dict[str, Any]:
        found: List[Dict[str, Any]] = []
        for pid, prim in self.lpe.primitives.items():
            try:
                pat = re.compile(prim.regex)
                matches = pat.findall(text)
                if matches:
                    found.append({"id": pid, "name": prim.name, "hits": len(matches)})
            except re.error:
                continue
        self.log.debug("  primitives matched: %d", len(found))
        return {"matched": len(found), "primitives": found}

    def _layer_gpl(self, text: str) -> Dict[str, Any]:
        try:
            res = self.gpl.traverse_lattice(text)
            self.log.debug("  lattice path: %s", " → ".join(res.get("path_taken", [])))
            return {"success": res["success"], "path": res["path_taken"], "captures": res.get("captures", {})}
        except Exception as exc:
            self.log.warning("  GPL traversal error: %s", exc)
            return {"success": False, "error": str(exc)}

    def _layer_scm(self, text: str) -> Dict[str, Any]:
        concepts = self.scm.extract_concepts_from_text(text)
        relations = self.scm.extract_relations_from_text(text)
        if concepts:
            self.scm.activate_concept(concepts[0].concept_id, 1.0)
        self.log.debug("  concepts=%d  relations=%d", len(concepts), len(relations))
        return {"concepts": len(concepts), "relations": len(relations)}

    def _layer_tam(self, text: str) -> Dict[str, Any]:
        try:
            mha = self.tam.multi_head_attention(text)
            self.log.debug("  heads fired: %s", mha["heads_used"])
            return mha
        except Exception as exc:
            self.log.warning("  TAM error: %s", exc)
            return {"heads_used": [], "concatenated_output": "", "error": str(exc)}

    def _layer_rge(self, text: str) -> Dict[str, Any]:
        trees: Dict[str, Any] = {}
        for rule in ("SYNTHOS_ROOT", "STATEMENT", "EXPRESSION"):
            tree = self.rge.parse(text, rule)
            if tree:
                trees[rule] = tree.to_dict()
        self.log.debug("  parse trees: %s", list(trees.keys()) or "(none)")
        return {"trees": trees, "matched_rules": list(trees.keys())}

    def _layer_scf(self, text: str, tam_out: Dict[str, Any]) -> Dict[str, Any]:
        # Crystallise first word
        words = text.split()
        if words:
            m = re.match(r"\w+", words[0])
            if m:
                self.scf.crystallize_match(r"\w+", m, LayerType.LEXICAL)

        coherence = self.scf.compute_coherence()
        self.log.debug("  coherence=%.3f  gate=%s", coherence, self.scf.apply_gate_conditions())
        return {"coherence": coherence, "gate_open": self.scf.apply_gate_conditions()}

    def _layer_ops(self, text: str, scf_out: Dict[str, Any]) -> str:
        # Try intent processing
        if "EMIT" in text.upper():
            try:
                processed = self.ops.process_input(text, "intent_to_language")
                if "RENDER_" in processed:
                    return self.ops.execute_render_command(processed)
                return processed
            except Exception:
                pass

        # Default structured response
        coh = scf_out.get("coherence", 0)
        return self.ops.render_template(TemplateType.PARAGRAPH, {
            "TOPIC": "SYNTHOS Response",
            "BODY": (
                f"Processed '{text[:60]}{'…' if len(text) > 60 else ''}' through 7 layers.  "
                f"Coherence {coh:.2f}.  "
                f"Memory: {len(self.scf.memory_lattice.short_term_buffer)} ST / "
                f"{len(self.scf.memory_lattice.long_term_registers)} LT / "
                f"{len(self.scf.memory_lattice.episodic_stack)} EP."
            ),
        })

    # ── Command handlers ──────────────────────────────────────────────────

    def _handle_command(self, cmd: str) -> str:
        parts = cmd.split(maxsplit=1)
        verb = parts[0].lower()
        arg = parts[1] if len(parts) > 1 else ""

        dispatch = {
            "/encrypt": self._cmd_encrypt,
            "/decrypt": self._cmd_decrypt,
            "/eval":    self._cmd_eval,
            "/status":  self._cmd_status,
            "/layers":  self._cmd_layers,
            "/config":  self._cmd_config,
            "/verbose": self._cmd_verbose,
            "/help":    self._cmd_help,
        }

        handler = dispatch.get(verb)
        if handler:
            return handler(arg)
        return f"Unknown command: {verb}  (try /help)"

    def _ensure_key(self) -> STCKey:
        if self._stc_key is None:
            self._stc_key = STCKey.generate()
            self.log.info("[crypto]Generated ephemeral STC-256 key[/crypto]")
        return self._stc_key

    def _cmd_encrypt(self, arg: str) -> str:
        if not arg:
            return "Usage: /encrypt <plaintext>"
        key = self._ensure_key()
        mode = STCMode(self.config.crypto.mode)
        cipher = SymbolicTopologyCipher(key, mode, verbose=(self.vlevel >= VerboseLevel.VERBOSE))
        ct = cipher.encrypt(arg.encode())
        lines = [
            f"Mode  : STC-{mode.value.upper()}-256",
            f"Key   : {key.hex()[:16]}…",
            f"Input : {len(arg)} bytes",
            f"Output: {len(ct)} bytes",
            f"Hex   : {ct.hex()}",
        ]
        if cipher.trace:
            lines.append("")
            lines.append("Trace:")
            lines.extend(f"  {t}" for t in cipher.trace[:20])
        return "\n".join(lines)

    def _cmd_decrypt(self, arg: str) -> str:
        if not arg:
            return "Usage: /decrypt <hex>"
        if self._stc_key is None:
            return "No key loaded.  Encrypt something first or load a key."
        try:
            envelope = bytes.fromhex(arg.strip())
        except ValueError:
            return "Invalid hex string."
        mode = STCMode(self.config.crypto.mode)
        cipher = SymbolicTopologyCipher(self._stc_key, mode, verbose=(self.vlevel >= VerboseLevel.VERBOSE))
        try:
            pt = cipher.decrypt(envelope)
            return f"Decrypted ({len(pt)} bytes): {pt.decode(errors='replace')}"
        except ValueError as exc:
            return f"Decryption failed: {exc}"

    def _cmd_eval(self, arg: str) -> str:
        from synthos.eval.bench import ModelBenchmark, ModelCard, EvalSuite, Provider
        cards: List[ModelCard] = []
        for ep in self.config.models:
            cards.append(ModelCard(
                name=ep.name,
                provider=Provider(ep.provider),
                base_url=ep.base_url,
                model_id=ep.model_id,
                ctx_len=ep.ctx_len,
                params_b=ep.params_b,
                quant=ep.quant,
                cost_per_mtok=ep.cost_per_mtok,
            ))
        bench = ModelBenchmark(cards, verbose=(self.vlevel >= VerboseLevel.VERBOSE))
        try:
            results = bench.run()
            return ModelBenchmark.leaderboard(results)
        except Exception as exc:
            return f"Eval error: {exc}\nEnsure an inference server is running at the configured endpoint."

    def _cmd_status(self, _arg: str) -> str:
        s = self.get_status()
        lines = ["SYNTHOS System Status", "=" * 40]
        for k, v in s.items():
            lines.append(f"  {k:<20} {v}")
        return "\n".join(lines)

    def _cmd_layers(self, _arg: str) -> str:
        layers = [
            ("L0  LPE", "Lexical Primitive Engine",    f"{len(self.lpe.primitives)} primitives"),
            ("L1  GPL", "Geometric Parse Lattice",     f"{len(self.gpl.cells)} cells"),
            ("L2  SCM", "Semantic Construct Manifold",  f"{len(self.scm.concepts)} concepts"),
            ("L3  TAM", "Topological Attention Mesh",   f"{len(self.tam.attention_heads)} heads"),
            ("L4  RGE", "Recursive Grammar Engine",     f"{len(self.rge.production_rules)} rules"),
            ("L5  SCF", "State Crystallization Field",  f"coh={self.scf.compute_coherence():.2f}"),
            ("L6  OPS", "Output Projection Surface",    f"{len(self.ops.substitution_chains)} chains"),
        ]
        lines = ["SYNTHOS Layer Diagnostics", "=" * 55]
        for code, name, stat in layers:
            lines.append(f"  {code}  {name:<30} {stat}")
        return "\n".join(lines)

    def _cmd_config(self, _arg: str) -> str:
        import yaml
        from dataclasses import asdict
        return yaml.dump(asdict(self.config), default_flow_style=False, sort_keys=False)

    def _cmd_verbose(self, arg: str) -> str:
        try:
            level = int(arg) if arg else (self.vlevel.value + 1) % 3
            self.vlevel = VerboseLevel(level)
            self.log = get_logger("synthos.assistant", self.vlevel)
            return f"Verbosity set to {self.vlevel.name} ({self.vlevel.value})"
        except (ValueError, KeyError):
            return "Usage: /verbose [0|1|2]"

    def _cmd_help(self, _arg: str) -> str:
        return (
            "SYNTHOS Commands\n"
            "═══════════════════════════════════════════\n"
            "  /help               Show this help\n"
            "  /status             System status overview\n"
            "  /layers             Layer diagnostics\n"
            "  /config             Print active configuration\n"
            "  /verbose [0|1|2]    Set verbosity (quiet/normal/verbose)\n"
            "  /encrypt <text>     Encrypt text with STC cipher\n"
            "  /decrypt <hex>      Decrypt STC hex envelope\n"
            "  /eval               Run model benchmark\n"
            "\n"
            "Anything else is processed through the 7-layer pipeline.\n"
            "Try: 'EMIT RESPONSE TYPE:EXPLAIN SUBJECT:attention DEPTH:2'\n"
        )
