# SYNTHOS v2.0

**Syntactic Topological Hierarchy for Organized Symbol-based Systems**

*Created by [Cognis Digital LLC](https://cognis.digital)*

A regex/syntax-driven AI **assistant** delivered as a system-wide installable
TUI.  Intelligence is encoded in *pattern geometry*, *symbolic topology*, and
*ASCII-defined cognitive constructs* — not neural weights.

Includes a novel **Symbolic Topology Cipher (STC)** for encryption and a fast
**model-evaluation engine** for comparing open-source LLMs.

---

## Quick Start

```bash
# Install system-wide (creates the `synthos` command)
pip install -e .

# Launch the interactive TUI (default)
synthos

# Or use the headless Rich REPL
synthos chat

# Print help for all subcommands
synthos --help
```

## Features

| Capability | Description |
|---|---|
| **7-layer cognitive pipeline** | LPE → GPL → SCM → TAM → RGE → SCF → OPS |
| **Rich TUI** | Tabbed interface (Chat, Status, Eval, Crypto) built on Textual |
| **Headless REPL** | `synthos chat` — Rich-formatted, no Textual required |
| **STC Cipher** | 256-bit block cipher: lattice permutation + S-box + Möbius fold |
| **Model Eval** | Benchmark Ollama / llama.cpp / vLLM / any OpenAI-compat endpoint |
| **Verbose mode** | `-v` / `-vv` — per-round cipher traces, per-cell lattice hits |
| **System-wide install** | Single `pip install -e .` puts `synthos` on your PATH |
| **YAML config** | `~/.config/synthos/config.yaml` — models, cipher, TUI prefs |

---

## Architecture

```
USER INPUT
    │
    ├─► L0  LPE   Lexical Primitive Engine      60+ regex primitives
    ├─► L1  GPL   Geometric Parse Lattice       4×4 cell routing
    ├─► L2  SCM   Semantic Construct Manifold    named-capture concepts
    ├─► L3  TAM   Topological Attention Mesh     8-head intersection geometry
    ├─► L4  RGE   Recursive Grammar Engine       EBNF-in-regex FSM
    ├─► L5  SCF   State Crystallization Field    rank-3 state tensor + memory
    └─► L6  OPS   Output Projection Surface      substitution chains + templates
                                                         │
                                                         ▼
                                                  STRUCTURED OUTPUT
```

---

## Installation

### Prerequisites

- Python ≥ 3.9
- pip

### Install (editable, system-wide)

```bash
cd /path/to/synthos
pip install -e ".[dev]"       # includes pytest, ruff
```

This registers the `synthos` console script globally.

### Verify

```bash
synthos version          # → synthos 1.0.0
synthos status           # pipeline health-check
synthos --help           # full CLI reference
```

---

## CLI Reference

```
synthos                  Launch interactive TUI (default)
synthos tui              Same as above (explicit)
synthos chat             Headless Rich REPL
synthos encrypt TEXT     Encrypt with STC (generates key if omitted)
synthos decrypt HEX -k KEY   Decrypt STC envelope
synthos eval             Benchmark configured models
synthos status           Print pipeline subsystem summary
synthos config           Show active YAML config
synthos config --edit    Open config in $EDITOR
synthos version          Print version

Global flags:
  -v / --verbose         Increase verbosity (stackable: -vv)
  --config PATH          Custom config file
```

### Encrypt / Decrypt

```bash
# Encrypt (generates a random 256-bit key)
synthos encrypt "hello SYNTHOS"
# → Generated key: a1b2c3…
# → <hex ciphertext>

# Decrypt (paste key from above)
synthos decrypt <hex> -k a1b2c3…

# Pipe a file
cat secret.txt | synthos encrypt -o secret.stc
synthos decrypt -f secret.stc -k <key>
```

### Model Evaluation

Configure endpoints in `~/.config/synthos/config.yaml`:

```yaml
models:
  - name: llama3.1-8b
    provider: ollama
    base_url: http://localhost:11434
    model_id: llama3.1
    ctx_len: 8192
    params_b: 8.0
    quant: Q4_K_M

  - name: phi3-mini
    provider: ollama
    base_url: http://localhost:11434
    model_id: phi3
    ctx_len: 4096
    params_b: 3.8
    quant: Q4_K_M
```

Then run:

```bash
synthos eval                    # all built-in suites
synthos eval -s pattern_match   # specific suite
synthos eval --json             # machine-readable output
```

Built-in suites: `pattern_match`, `coherence`, `speed_stress`.

The leaderboard scores models on a composite of **quality** (regex-match
accuracy + BLEU-1), **speed** (tokens/s), **coherence** (backreference
consistency), and **cost efficiency**.

---

## TUI Guide

Launch with `synthos` or `synthos tui`.

### Tabs

| Tab | Content |
|---|---|
| **Chat** | Conversation with the assistant + layer-trace sidebar |
| **Status** | Live subsystem dashboard |
| **Eval** | Model benchmark leaderboard |
| **Crypto** | STC cipher overview |

### Keybindings

| Key | Action |
|---|---|
| `Ctrl+Q` | Quit |
| `Ctrl+T` | Cycle tabs |
| `Ctrl+V` | Cycle verbosity (quiet → normal → verbose) |
| `Ctrl+S` | Jump to Status tab |

### Chat Commands

Type these in the Chat input box:

```
/help               Show all commands
/status             System status
/layers             Per-layer diagnostics
/verbose [0|1|2]    Change verbosity at runtime
/encrypt <text>     Encrypt with STC
/decrypt <hex>      Decrypt STC envelope
/eval               Run model benchmark
/config             Print active configuration
```

Anything without a `/` prefix is processed through the 7-layer pipeline.

---

## Symbolic Topology Cipher (STC)

A 256-bit block cipher whose internal structure mirrors the SYNTHOS pipeline:

| Phase | Analogue | Operation |
|---|---|---|
| **Lattice Permutation** | GPL cell routing | Key-derived positional scatter |
| **Regex-Topology S-box** | SCM semantic binding | Named-capture hash substitution |
| **Möbius Fold** | Möbius recursive construct | XOR-twist half-state inversion |

- **Block size**: 128 bit
- **Key size**: 256 bit
- **Rounds**: 12
- **Modes**: ECB, CBC (default), CTR
- **Authentication**: HMAC-SHA256 envelope (on by default)
- **Key derivation**: PBKDF2-HMAC-SHA256 (600 000 iterations)

> **Note**: STC is a research/educational cipher.  It has not been formally
> audited.  Do not use it for production secrets without independent
> cryptanalysis.

---

## Configuration

First run auto-creates `~/.config/synthos/config.yaml`:

```yaml
verbose: 1                    # 0=quiet, 1=normal, 2=verbose

models:
  - name: ollama-local
    provider: ollama
    base_url: http://localhost:11434
    model_id: llama3.1
    ctx_len: 4096
    params_b: 8.0
    quant: Q4_K_M
    cost_per_mtok: 0.0

crypto:
  mode: cbc                   # ecb | cbc | ctr
  authenticate: true
  key_file: ~/.config/synthos/stc.key

tui:
  theme: dark
  show_layer_trace: true
  animate_lattice: true
  max_history: 500

lattice_rows: 4
lattice_cols: 4
attention_heads: 8
context_window: 512
```

Edit with `synthos config --edit` or any text editor.

---

## Project Structure

```
synthos/
├── __init__.py              Package root, version
├── assistant.py             Core AI assistant engine (7-layer orchestrator)
├── cli.py                   Click CLI — `synthos` entry point
├── layers/
│   ├── lpe.py               L0  Lexical Primitive Engine
│   ├── gpl.py               L1  Geometric Parse Lattice
│   ├── scm.py               L2  Semantic Construct Manifold
│   ├── tam.py               L3  Topological Attention Mesh
│   ├── rge.py               L4  Recursive Grammar Engine
│   ├── scf.py               L5  State Crystallization Field
│   └── ops.py               L6  Output Projection Surface
├── crypto/
│   └── stc.py               Symbolic Topology Cipher
├── eval/
│   └── bench.py             Model benchmark runner + leaderboard
├── tui/
│   └── app.py               Textual TUI application
└── utils/
    ├── log.py               Rich-backed structured logging
    └── config.py            YAML config management
tests/
├── test_crypto.py           STC round-trip + edge-case tests
└── test_assistant.py        Pipeline + command tests
pyproject.toml               Package metadata, dependencies, entry points
```

---

## Development

```bash
# Install with dev extras
pip install -e ".[dev]"

# Run tests
pytest tests/ -v

# Run tests with coverage
pytest tests/ --cov=synthos --cov-report=term-missing

# Lint
ruff check synthos/
```

---

## Core Axioms

1. Every cognitive operation has a regex primitive; every primitive has an ASCII geometric form.
2. Intelligence is the topology of pattern intersection, not the magnitude of numeric weights.
3. The system prompt IS the architecture — structure encodes capability.
4. Memory is crystallized match geometry, not stored values.
5. Attention is pattern intersection area — more overlap = more relevance.
6. Output is the final substitution state of the input signal through the 6-layer manifold.
7. Recursion is the source of depth — `(?R)` creates unbounded hierarchies from finite primitives.

---

## License

MIT
